/*
 * Copyright (c) 2007, Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, v. 2.1. This program is distributed in the
 * hope that it will be useful, but WITHOUT A WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details. You should have received a
 * copy of the GNU Lesser General Public License, v.2.1 along with this
 * distribution; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Red Hat Author(s): Steve Ebersole
 */
package org.jboss.maven.plugins.docbook.support;

import java.io.File;
import java.io.IOException;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.jboss.maven.shared.file.DirectoryCopier;

/**
 * Defines resources-like processing for the jdocbook-style packaging.  Handles
 * files such as XSLT, fonts, images and css, moving them to specific locations
 * within the output directory for inclusion in the final package.
 *
 * @goal resources
 * @phase process-resources
 * @requiresDependencyResolution
 *
 * @author Steve Ebersole
 */
public class ResourceMojo extends AbstractMojo {

	/**
	 * The directory containing the XSLT sources.
	 *
	 * @parameter expression="${basedir}/src/main/xslt"
	 */
	protected File xsltSourceDirectory;

	/**
	 * The directory containing fonts to be included in package.
	 *
	 * @parameter expression="${basedir}/src/main/fonts"
	 */
	protected File fontSourceDirectory;

	/**
	 * The directory containing images to be included in package.
	 *
	 * @parameter expression="${basedir}/src/main/images"
	 */
	protected File imagesSourceDirectory;

	/**
	 * The directory containing css to be included in package.
	 *
	 * @parameter expression="${basedir}/src/main/css"
	 */
	protected File cssSourceDirectory;

	/**
	 * The directory from which packaging is staged.
	 *
	 * @parameter expression="${project.build.outputDirectory}"
	 */
	protected File outputDirectory;

	public void execute() throws MojoExecutionException, MojoFailureException {
		copySource( xsltSourceDirectory, new File( outputDirectory, "xslt" ) );
		copySource( fontSourceDirectory, new File( outputDirectory, "fonts" ) );
		copySource( imagesSourceDirectory, new File( outputDirectory, "images" ) );
		copySource( cssSourceDirectory, new File( outputDirectory, "css" ) );
	}

	private void copySource(File sourceDirectory, File targetDirectory) throws MojoExecutionException {
		getLog().info( "attempting to copy directory : " + sourceDirectory.getAbsolutePath() );
		try {
			new DirectoryCopier( sourceDirectory ).copyTo( targetDirectory );
		}
		catch ( IOException e ) {
			throw new MojoExecutionException( "Unable to copy directory [" + sourceDirectory + "]", e );
		}
	}
}