/*
 * Copyright (c) 2007, Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, v. 2.1. This program is distributed in the
 * hope that it will be useful, but WITHOUT A WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details. You should have received a
 * copy of the GNU Lesser General Public License, v.2.1 along with this
 * distribution; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Red Hat Author(s): Steve Ebersole
 */
package org.jboss.maven.shared.file;

import java.io.File;
import java.io.IOException;

import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;

/**
 * Copies a directory with its structure intact.  Optionally allows defining
 * includes and/or excludes.  Internally uses a {@link DirectoryScanner}, mainly
 * to ensure we filter out "standard excludes" (like scm dirs) but also to apply
 * any optionl includes and/or excludes.
 *
 * @author Steve Ebersole
 */
public class DirectoryCopier {
	private final File sourceDirectory;

	public DirectoryCopier(File sourceDirectory) {
		this.sourceDirectory = sourceDirectory;
	}

	public void copyTo(File targetDirectory) throws IOException {
		copyTo( targetDirectory, null, null );
	}

	private void copyTo(File targetDirectory, String[] includes, String[] excludes) throws IOException {
		if ( !sourceDirectory.exists() ) {
			return;
		}
		String[] list = sourceDirectory.list();
		if ( list == null || list.length == 0 ) {
			return;
		}

		if ( !targetDirectory.exists() ) {
			targetDirectory.mkdirs();
		}

		DirectoryScanner scanner = new DirectoryScanner();
		scanner.setBasedir( sourceDirectory );
		scanner.setIncludes( includes );
		scanner.setExcludes( excludes );
		scanner.addDefaultExcludes();
		scanner.scan();
		String[] includedFiles = scanner.getIncludedFiles();

		for ( int i = 0; i < includedFiles.length; i++ ) {
			copyFileIfModified(
					new File( sourceDirectory, includedFiles[i] ),
					new File( targetDirectory, includedFiles[i] )
			);
		}
	}

	public static void copyFileIfModified(File source, File destination) throws IOException {
		if ( destination.lastModified() < source.lastModified() ) {
			FileUtils.copyFile( source.getCanonicalFile(), destination );
			destination.setLastModified( source.lastModified() );
		}
	}

}
