/*
 * Copyright (c) 2007, Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, v. 2.1. This program is distributed in the
 * hope that it will be useful, but WITHOUT A WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details. You should have received a
 * copy of the GNU Lesser General Public License, v.2.1 along with this
 * distribution; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Red Hat Author(s): Steve Ebersole
 */
package org.jboss.maven.plugins.test.ext;

import java.util.ArrayList;
import java.util.Iterator;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Exclusion;
import org.apache.maven.model.Resource;
import org.dom4j.Element;
import org.jboss.maven.shared.xml.dom4j.AttributeValueExtracter;
import org.jboss.maven.shared.xml.dom4j.Extracter;
import org.jboss.maven.shared.xml.dom4j.ElementValueExtracter;
import org.jboss.maven.shared.xml.dom4j.NestedElementValueExtracter;

/**
 * Builds {@link Environment} instances by parsing its XML definition.
 *
 * @author Steve Ebersole
 */
public class EnvironmentBuilder {

	public static final String ENVIRONMENT = "environment";

	private static Extracter ENV_NAME_EXTRACTER = new AttributeValueExtracter( "name", true, null );

	private static Extracter BASIC_ELEMENT_VALUE_EXTRACTER = new ElementValueExtracter();

	private static Extracter GROUPID_EXTRACTER = new NestedElementValueExtracter( "groupId", true, null );
	private static Extracter ARTIFACTID_EXTRACTER = new NestedElementValueExtracter( "artifactId", true, null );
	private static Extracter VERSION_EXTRACTER = new NestedElementValueExtracter( "version", true, null );
	private static Extracter TYPE_EXTRACTER = new NestedElementValueExtracter( "type", false, "jar" );
	private static Extracter CLASSIFIER_EXTRACTER = new NestedElementValueExtracter( "classifier", false, null );
	private static Extracter SYSPATH_EXTRACTER = new NestedElementValueExtracter( "systemPath", false, null );
	private static Extracter OPTIONAL_EXTRACTER = new NestedElementValueExtracter( "optional", false, "false" );

	private static Extracter DIRECTORY_EXTRACTER = new NestedElementValueExtracter( "directory", true, null );
	private static Extracter TARGETPATH_EXTRACTER = new NestedElementValueExtracter( "targetPath", false, null );
	private static Extracter FILTERING_EXTRACTER = new NestedElementValueExtracter( "filtering", false, null );

	/**
	 * Constructs a new EnvironmentBuilder instance.
	 */
	public EnvironmentBuilder() {
	}

	public String extractEnvironmentName(Element environmentElement) {
		return ENV_NAME_EXTRACTER.extract( environmentElement );
	}

	public Environment buildEnvironment(Element environmentElement) {
		return extractEnvironmentInfo( environmentElement );
	}

	private Environment extractEnvironmentInfo(Element environmentElement) {
		Environment env = new Environment( extractEnvironmentName( environmentElement ) );

		// deps...
		Element dependenciesElement = environmentElement.element( "dependencies" );
		if ( dependenciesElement != null ) {
			Iterator dependencyElements = dependenciesElement.elementIterator( "dependency" );
			while ( dependencyElements.hasNext() ) {
				final Element dependencyElement = ( Element ) dependencyElements.next();
				env.getDependencies().add( extractDependencyInfo( dependencyElement ) );
			}
		}

		// resources...
		Element resourcesElement = environmentElement.element( "testResources" );
		if ( resourcesElement != null ) {
			Iterator resourceElements = resourcesElement.elementIterator( "testResource" );
			while ( resourceElements.hasNext() ) {
				final Element resourceElement = ( Element ) resourceElements.next();
				env.getResources().add( extractResourceInfo( resourceElement ) );
			}
		}

		return env;
	}

	private Dependency extractDependencyInfo(Element dependencyElement) {
		Dependency dependency = new Dependency();
		dependency.setModelEncoding( dependencyElement.getDocument().getXMLEncoding() );
		dependency.setGroupId( GROUPID_EXTRACTER.extract( dependencyElement ) );
		dependency.setArtifactId( ARTIFACTID_EXTRACTER.extract( dependencyElement ) );
		dependency.setVersion( VERSION_EXTRACTER.extract( dependencyElement ) );
		dependency.setType( TYPE_EXTRACTER.extract( dependencyElement ) );
		dependency.setClassifier( CLASSIFIER_EXTRACTER.extract( dependencyElement ) );
		dependency.setSystemPath( SYSPATH_EXTRACTER.extract( dependencyElement ) );
		dependency.setOptional( Boolean.valueOf( OPTIONAL_EXTRACTER.extract( dependencyElement ) ).booleanValue() );
		// ignore scope in environment, but instead set it here to test for correctness
		dependency.setScope( "test" );

		ArrayList exclusions = new ArrayList();
		Element exclusionsElement = dependencyElement.element( "exclusions" );
		if ( exclusionsElement != null ) {
			Iterator itr = exclusionsElement.elementIterator( "exclusion" );
			while ( itr.hasNext() ) {
				final Element exclusionElement = ( Element ) itr.next();
				exclusions.add( extractExclusionInfo( exclusionElement ) );
			}
		}
		dependency.setExclusions( exclusions );

		return dependency;
	}

	private Object extractExclusionInfo(Element exclusionElement) {
		Exclusion exclusion = new Exclusion();
		exclusion.setModelEncoding( exclusionElement.getDocument().getXMLEncoding() );
		exclusion.setGroupId( GROUPID_EXTRACTER.extract( exclusionElement ) );
		exclusion.setArtifactId( ARTIFACTID_EXTRACTER.extract( exclusionElement ) );
		return exclusion;
	}

	private Resource extractResourceInfo(Element resourceElement) {
		Resource resource = new Resource();
		resource.setModelEncoding( resourceElement.getDocument().getXMLEncoding() );
		resource.setDirectory( DIRECTORY_EXTRACTER.extract( resourceElement ) );
		resource.setTargetPath( TARGETPATH_EXTRACTER.extract( resourceElement ) );
		resource.setFiltering( Boolean.valueOf( FILTERING_EXTRACTER.extract( resourceElement ) ).booleanValue() );

		ArrayList includes = new ArrayList();
		Element includesElement = resourceElement.element( "includes" );
		if ( includesElement != null ) {
			Iterator itr = includesElement.elementIterator( "include" );
			while ( itr.hasNext() ) {
				final Element includeElement = ( Element ) itr.next();
				includes.add( BASIC_ELEMENT_VALUE_EXTRACTER.extract( includeElement ) );
			}
		}
		resource.setIncludes( includes );

		ArrayList excludes = new ArrayList();
		Element excludesElement = resourceElement.element( "excludes" );
		if ( excludesElement != null ) {
			Iterator itr = excludesElement.elementIterator( "exclude" );
			while ( itr.hasNext() ) {
				final Element excludeElement = ( Element ) itr.next();
				excludes.add( BASIC_ELEMENT_VALUE_EXTRACTER.extract( excludeElement ) );
			}
		}
		resource.setExcludes( excludes );

		return resource;
	}

}
