/*
 * Copyright (c) 2007, Red Hat Middleware, LLC. All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, v. 2.1. This program is distributed in the
 * hope that it will be useful, but WITHOUT A WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details. You should have received a
 * copy of the GNU Lesser General Public License, v.2.1 along with this
 * distribution; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Red Hat Author(s): Steve Ebersole
 */
package org.jboss.maven.plugins.test.ext;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.AbstractArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactCollector;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolutionResult;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.resolver.DebugResolutionListener;
import org.apache.maven.artifact.resolver.ResolutionNode;
import org.apache.maven.artifact.resolver.WarningResolutionListener;
import org.apache.maven.artifact.versioning.InvalidVersionSpecificationException;
import org.apache.maven.artifact.versioning.VersionRange;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.logging.LogEnabled;
import org.codehaus.plexus.logging.Logger;
import org.dom4j.Document;
import org.dom4j.Element;
import org.jboss.maven.shared.xml.dom4j.DocumentLoader;

/**
 * Extends the test environment by expanding the test classpath based on some
 * external configuration.  Mainly this is useful in integration testing
 * scenarios where you want to allow end-users (testers) to supply dependencies
 * (jdbc drivers, e.g.) and/or config (properties).
 * <p/>
 * The accepted external configuration format is described by the DTD in this
 * plugins resources (extend.dtd)
 *
 * @goal extend
 * @phase generate-resources
 * @requiresDependencyResolution
 *
 * @author Steve Ebersole
 */
public class ExtenderMojo extends AbstractMojo implements LogEnabled {

	public static final String FILE_SEPARATOR = "/";

	/**
	 * INTERNAL : The Maven project
	 *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    protected MavenProject project;

	/**
     * INTERNAL : Local maven repository.
     *
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     */
    protected ArtifactRepository localRepository;

	/**
     * INTERNAL : Artifact collector, needed to resolve dependencies.
     *
     * @component role="org.apache.maven.artifact.resolver.ArtifactCollector"
     * @required
     * @readonly
     */
    protected ArtifactCollector artifactCollector;

	/**
	 * INTERNAL : Artifact factory, needed to download dependencies
	 *
	 * @component role="org.apache.maven.artifact.factory.ArtifactFactory"
	 * @required
	 * @readonly
	 */
	protected ArtifactFactory artifactFactory;

	/**
	 * INTERNAL : Artifact resolver, needed to download dependencies
	 *
	 * @component role="org.apache.maven.artifact.resolver.ArtifactResolver"
	 * @required
	 * @readonly
	 */
	protected ArtifactResolver artifactResolver;

	/**
	 * INTERNAL : The artifact metadata source ;)
	 *
     * @component role="org.apache.maven.artifact.metadata.ArtifactMetadataSource" hint="maven"
     */
    protected ArtifactMetadataSource artifactMetadataSource;

	/**
	 * The path to the extender configuration file to use.  Defined as String
	 * so that it can be specified as a build property
	 *
	 * @parameter expression="${test.extender.cfg}"
	 */
	protected String extenderConfig;

	/**
	 * As a matter of convenience, an extender config can name multiple 
	 * environment entries.  However, for a test run a particular one must
	 * be selected from the many.  This parameter specifies which environment
	 * to use.
	 * <p/>
	 * The anticipated usage here is to set this as a build property...
	 *
	 * @parameter expression="${test.extender.env}"
	 */
	protected String extenderEnv;


	// LogEnabled impl ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	private Logger logger;

	public void enableLogging(Logger logger) {
		this.logger = logger;
	}


	// Mojo#execute impl ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	public void execute() throws MojoExecutionException, MojoFailureException {
		getLog().info( "starting test environment extension mojo" );
		Environment environment = parseEnvironment();
		extendTestClasspath( environment );
	}


	// read/parse XML ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	protected Environment parseEnvironment() throws MojoExecutionException {
		Document extenderConfigDocument = loadExtenderConfigDocument();
		EnvironmentBuilder environmentBuilder = new EnvironmentBuilder();
		Iterator itr = extenderConfigDocument.getRootElement().elementIterator( EnvironmentBuilder.ENVIRONMENT );
		while ( itr.hasNext() ) {
			final Element environmentElement = ( Element ) itr.next();
			if ( extenderEnv.equals( environmentBuilder.extractEnvironmentName( environmentElement ) ) ) {
				// EARLY EXIT!!!!!
				getLog().info( "found environment definition [" + extenderEnv + "]" );
				return environmentBuilder.buildEnvironment( environmentElement );
			}
		}
		getLog().info( "Unable to locate appropriate extender env [" + extenderEnv + "] in specified config [" + extenderConfig + "]" );
		return null;
	}

	protected Document loadExtenderConfigDocument() throws MojoExecutionException {
		File extenderConfigFile = new File( extenderConfig );
		if ( !extenderConfigFile.exists() ) {
			getLog().warn( "Could not locate specified extender config file [" + extenderConfig + "]" );
		}

		return new DocumentLoader( getLog() ).loadDocument( extenderConfigFile );
	}


	// extend classpath ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	protected void extendTestClasspath(Environment environment) throws MojoExecutionException {
		attachDependencies( environment.getDependencies() );
		attachResources( environment.getResources() );
	}

	protected void attachDependencies(List dependencies) throws MojoExecutionException {
		Set combinedDependencyArtifacts = new HashSet();
		combinedDependencyArtifacts.addAll( project.getDependencyArtifacts() );

		Set artifacts = new HashSet();
		Iterator itr = dependencies.iterator();
		while ( itr.hasNext() ) {
			final Dependency dependency = ( Dependency ) itr.next();
			try {
				VersionRange versionRange = VersionRange.createFromVersionSpec( dependency.getVersion() );
				Artifact artifact = artifactFactory.createDependencyArtifact(
						dependency.getGroupId(),
						dependency.getArtifactId(),
						versionRange,
						dependency.getType(),
						dependency.getClassifier(),
						Artifact.SCOPE_TEST,
						false
				);
				artifacts.add( artifact );
			}
			catch( InvalidVersionSpecificationException e ) {
				throw new MojoExecutionException( "Unable to parse version" );
			}
		}


		try {
			List listeners = new ArrayList();
			listeners.add( new WarningResolutionListener( logger ) );
			if ( logger.isDebugEnabled() ) {
				listeners.add( new DebugResolutionListener( logger ) );
			}

			ArtifactResolutionResult artifactResolutionResult = artifactCollector.collect(
					artifacts,
					project.getArtifact(),
					localRepository, 
					project.getRemoteArtifactRepositories(),
					artifactMetadataSource,
					null,
					listeners
			);

			itr = artifactResolutionResult.getArtifactResolutionNodes().iterator();
			while ( itr.hasNext() ) {
				final ResolutionNode node = ( ResolutionNode ) itr.next();
				final Artifact artifact = node.getArtifact();
				if ( ! artifact.isResolved() ) {
					artifactResolver.resolve( artifact, node.getRemoteRepositories(), localRepository );
				}
				getLog().info( "adding dependency artifact [" + extenderEnv + "] : " + artifact.getId() );
			}

		}
		catch( ArtifactNotFoundException e ) {
			throw new MojoExecutionException( "Unable to download artifact " + buildInfo( e ) + " : " + e.getMessage(), e );
		}
		catch ( ArtifactResolutionException e ) {
			throw new MojoExecutionException( "Unable to resolve artifact " + buildInfo( e ) + " : " + e.getMessage(), e );
		}

		combinedDependencyArtifacts.addAll( artifacts );
		project.setDependencyArtifacts( combinedDependencyArtifacts );
		project.setArtifacts( null );
	}

	protected void attachResources(List resources) {
		Iterator itr = resources.iterator();
		while ( itr.hasNext() ) {
			final Resource resource = ( Resource ) itr.next();
			alignToBaseDirectory( resource );
			project.addTestResource( resource );
		}
	}

	private void alignToBaseDirectory(Resource resource) {
		resource.setDirectory( alignToBaseDirectory( resource.getDirectory(), project.getBasedir() ) );
	}

	public String alignToBaseDirectory(String path, File basedir) {
		String s = stripBasedirToken( path );

		if ( requiresBaseDirectoryAlignment( s ) ) {
			s = new File( basedir, s ).getAbsolutePath();
		}

		return s;
	}

	private String stripBasedirToken(String s) {
		if ( s != null ) {
			s = s.trim();

			if ( s.startsWith( "${basedir}" ) ) {
				// Take out ${basedir} and the leading slash
				s = s.substring( 11 );
			}
		}

		return s;
	}

	private boolean requiresBaseDirectoryAlignment(String s) {
		if ( s != null ) {
			File f = new File( s );
			return !( s.startsWith( FILE_SEPARATOR ) || f.isAbsolute() );
		}

		return false;
	}

	private String buildInfo(AbstractArtifactResolutionException e) {
		return "[" + e.getGroupId() + ":" + e.getArtifactId() + ":" + e.getVersion() + "]";
	}
}
