/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.mcann.indexer;

import org.jboss.mcann.AnnotationRepository;
import org.jboss.mcann.AnnotationScanner;
import org.jboss.mcann.AnnotationScannerFactory;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.StringTokenizer;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

/**
 * The Papaki Indexer Ant task
 * @author <a href="mailto:jesper.pedersen@jboss.org">Jesper Pedersen</a>
 */
public class IndexerTask extends Task
{
   private File input;
   private File output;
   private boolean update;
   private File metadata;
   private boolean excludeAll;
   private String exclude;
   private File report;
   
   /**
    * Constructor
    */
   public IndexerTask()
   {
      input = null;
      output = null;
      update = true;
      metadata = null;
      excludeAll = false;
      exclude = null;
      report = null;
   }

   /**
    * Get the input file
    * @return The file
    */
   public File getInput()
   {
      return input;
   }

   /**
    * Set the input file
    * @param f The file
    */
   public void setInput(File f)
   {
      input = f;
   }

   /**
    * Get the output file
    * @return The file
    */
   public File getOutput()
   {
      return output;
   }

   /**
    * Set the output file
    * @param f The file
    */
   public void setOutput(File f)
   {
      output = f;
   }

   /**
    * Get the update mode
    * @return The mode
    */
   public boolean isUpdate()
   {
      return update;
   }

   /**
    * Set the update mode
    * @param b The mode
    */
   public void setUpdate(boolean b)
   {
      update = b;
   }

   /**
    * Get the metadata
    * @return The metadata
    */
   public File getMetadata()
   {
      return metadata;
   }

   /**
    * Set the metadata
    * @param f The metadata
    */
   public void setMetadata(File f)
   {
      metadata = f;
   }

   /**
    * Get the exclude
    * @return The exclude
    */
   public String getExclude()
   {
      return exclude;
   }

   /**
    * Get the exclude all mode
    * @return The mode
    */
   public boolean isExcludeAll()
   {
      return excludeAll;
   }

   /**
    * Set the exclude all mode
    * @param b The mode
    */
   public void setExcludeAll(boolean b)
   {
      excludeAll = b;
   }

   /**
    * Set the exclude
    * @param s The exclude
    */
   public void setExclude(String s)
   {
      exclude = s;
   }
 
   /**
    * Get the report directory
    * @return The report directory
    */
   public File getReport()
   {
      return report;
   }

   /**
    * Set the report directory
    * @param r The report directory
    */
   public void setReport(File r)
   {
      report = r;
   }

  /**
    * Execute Ant task
    * @exception BuildException If an error occurs
    */
   public void execute() throws BuildException
   {
      try
      {
         if (output == null)
            output = input;

         Main m = new Main();
            
         AnnotationScanner as = AnnotationScannerFactory.getStrategy(AnnotationScannerFactory.JAVA_LANG_REFLECT);
         URL[] urls = {input.toURI().toURL()};
         URLClassLoader ucl = new URLClassLoader(urls);
         AnnotationRepository ari = as.scan(ucl, urls);

         if (excludeAll)
         {
            // TODO ari.removeAllAnnotations();
         }
         else
         {
            if (exclude != null)
            {
               StringTokenizer st = new StringTokenizer(exclude, ",");
               while (st.hasMoreTokens())
               {
                  String s = st.nextToken();
                  s = s.trim();

                  if (s.endsWith(".class"))
                     s = s.substring(0, s.indexOf(".class"));
                  
                  // TODO ari.removeAnnotations(s);
               }
            }
         }

         if (report != null)
         {
            if (!report.exists())
            {
               if (!report.mkdirs())
                  throw new IOException("Couldn't create directory: " + report);
            }
            else
            {
               if (!report.isDirectory())
                  throw new IOException(report + " is not a directory");
            }

            HTMLWriter.generateCSS(report);
            // HTMLWriter.generateReport(report, output.getName(), ari.getAnnotationToClasses());
         }

         if (update)
         {
            File tmp = new File(System.getProperty("java.io.tmpdir"));

            File root = FileUtil.extract(input, tmp);
            File destination = new File(root, "META-INF");

            m.store(ari, destination);

            FileUtil.compress(root, output);
               
            FileUtil.recursiveDelete(root);
         }
         else
         {
            if (metadata.exists())
            {
               if (metadata.isFile())
                  throw new IOException("Metadata should be a directory: " + metadata);
            }
            else
            {
               if (!metadata.mkdirs())
                  throw new IOException("Unable to create directory: " + metadata);
            }

            m.store(ari, metadata);
         }
      }
      catch (Throwable t)
      {
         throw new BuildException(t.getMessage(), t);
      }
   }
}
