/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.mx.notification;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;

import javax.management.JMException;
import javax.management.ListenerNotFoundException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;

/**
 * A notification listener registry.<p>
 *
 * For addition and removal, the registrations are deeply cloned to 
 * allow the registrations to be iterated externally without 
 * incurring the cost of synchronization.
 * 
 * @see org.jboss.mx.notification.ListenerRegistration
 * 
 * @author  <a href="mailto:Adrian.Brock@HappeningTimes.com">Adrian Brock</a>.
 * @author  <a href="mailto:juha@jboss.org">Juha Lindfors</a>.
 *
 * @version $Revision: 85671 $
 *
 * <p><b>Revisions:</b>
 *
 * <p><b>20030806 Juha Lindfors:</b>
 * <ul>
 * <li>
 *   Added removeAll()
 * </ul>
 *
 */
public class ListenerRegistry
{
   // Attributes ----------------------------------------------------

   /**
    * A map of listeners to a list of NotificationRegistrations.
    */
   private HashMap<NotificationListener, ArrayList<ListenerRegistration>> listeners = new HashMap<NotificationListener, ArrayList<ListenerRegistration>>();

   /**
    * The factory used to generate the listener registration.
    */
   private ListenerRegistrationFactory factory;


   // Constructor ---------------------------------------------------

   /**
    * Create a notification listener registry using the default
    * listener registration factory.
    */
   public ListenerRegistry()
   {
      this(null);
   }

   /**
    * Create a notification listener registry using the passed
    * listener registration factory.<p>
    *
    * @param factory the factory to create registrations, use
    *        null for the default factory
    */
   public ListenerRegistry(ListenerRegistrationFactory factory)
   {
      if (factory == null)
         this.factory = new DefaultListenerRegistrationFactory();
      else
         this.factory = factory;
   }

   // Public --------------------------------------------------------
   
   /**
    * Adds a listener to a broadcaster<p>
    *
    * @param listener the listener to register
    * @param filter filters the notifications in the broadcaster, can be null
    * @param handback the object to include in the notification, can be null
    * @exception IllegalArgumentException for a null listener
    * @exception JMException for an error adding to the registry
    */
   @SuppressWarnings("unchecked")
   public void add(NotificationListener listener, NotificationFilter filter, Object handback)
      throws JMException
   {
      if (listener == null)
         throw new IllegalArgumentException("Null listener");

      synchronized(listeners)
      {
         HashMap<NotificationListener, ArrayList<ListenerRegistration>> newListeners = cloneListeners();

         ArrayList<ListenerRegistration> registrations = newListeners.get(listener);
         if (registrations == null)
         {
            registrations = new ArrayList<ListenerRegistration>();
            newListeners.put(listener, registrations);
         }
         else
         {
            registrations = (ArrayList) registrations.clone();
            newListeners.put(listener, registrations);
         }

         registrations.add(factory.create(listener, filter, handback));

         listeners = newListeners;
      }
   }

   /**
    * Removes all registrations for a listener.
    *
    * @param listener the listener to remove
    * @exception ListenerNotFoundException when the listener is not registered
    */
   public void remove(NotificationListener listener)
      throws ListenerNotFoundException
   {
      ArrayList<ListenerRegistration> registrations = null;
      synchronized(listeners)
      {
         if (listeners.containsKey(listener) == false)
            throw new ListenerNotFoundException("Listener not found " + listener);

         HashMap<NotificationListener, ArrayList<ListenerRegistration>> newListeners = cloneListeners();

         registrations = newListeners.remove(listener);

         listeners = newListeners;
      }

      for (ListenerRegistration registration : registrations)
         registration.removed();
   }

   /**
    * Removes only the registrations for a listener that match the filter and handback.
    *
    * @param listener the listener to remove
    * @param filter the filter of the registration to remove
    * @param handback the handback object of the registration to remove
    * @exception ListenerNotFoundException when the listener is not registered
    */
   @SuppressWarnings("unchecked")
   public void remove(NotificationListener listener, NotificationFilter filter, Object handback)
      throws ListenerNotFoundException
   {
      ListenerRegistration registration = null;
      synchronized(listeners)
      {
         ArrayList<ListenerRegistration> registrations = listeners.get(listener);
         if (registrations == null)
            throw new ListenerNotFoundException("No registristrations for listener not listener=" + listener +
                                                " filter=" + filter + " handback=" + handback);

         registration = new DefaultListenerRegistration(listener, filter, handback);
         int index = registrations.indexOf(registration);
         if (index == -1)
            throw new ListenerNotFoundException("Listener not found listener=" + listener +
                                                " filter=" + filter + " handback=" + handback);

         HashMap<NotificationListener, ArrayList<ListenerRegistration>> newListeners = cloneListeners();

         registrations = (ArrayList) registrations.clone();
         registration = registrations.remove(index);
         if (registrations.isEmpty())
            newListeners.remove(listener);
         else
            newListeners.put(listener, registrations);

         listeners = newListeners;
      }

      registration.removed();
   }

   /**
    *  Removes all listeners from this listener registry.
    */
   public void removeAll()
   {
      synchronized (listeners)
      {
         listeners.clear();
      }
   }

   /**
    * Retrieve an iterator over the registrations<p>
    *
    * The iterator behaves like a snapshot of the registrations
    * is taken during this operation.
    *
    * @return the iterator
    */
   public ListenerRegistrationIterator iterator()
   {
      return new ListenerRegistrationIterator();
   }

   /**
    * Test whether the registry is empty
    *
    * @return true when it is empty, false otherwise
    */
   public boolean isEmpty()
   {
      return listeners.isEmpty();
   }

   @SuppressWarnings("unchecked")
   private HashMap<NotificationListener, ArrayList<ListenerRegistration>> cloneListeners()
   {
      return (HashMap) listeners.clone();
   }

   // Inner Classes -------------------------------------------------

   public class ListenerRegistrationIterator
      implements Iterator<ListenerRegistration>
   {
      private Iterator<ArrayList<ListenerRegistration>> listenerIterator;
      private Iterator<ListenerRegistration> registrationIterator;

      /**
       * Constructs a new ListenerRegistration iterator
       */
      public ListenerRegistrationIterator()
      {
         listenerIterator = listeners.values().iterator();
      }

      public boolean hasNext()
      {
         if (registrationIterator == null || registrationIterator.hasNext() == false)
         {
            do
            {
               if (listenerIterator.hasNext() == false)
                  return false;

               registrationIterator = listenerIterator.next().iterator();
            }
            while (registrationIterator.hasNext() == false);
         }
         return true;
      }

      public ListenerRegistration next()
      {
         if (hasNext() == false)
            throw new NoSuchElementException("Use hasNext before next");

         return registrationIterator.next();
      }

      /**
       * Convenience method to returned a typed object
       * 
       * @return the registration
       */
      public ListenerRegistration nextRegistration()
      {
         return next();
      }

      /**
       * @exception UnsupportedOperationException remove is not supported
       */
      public void remove()
      {
         throw new UnsupportedOperationException("remove is not supported");
      }
   }
}
