/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.container;

//$Id: BlueprintContextImpl.java 90313 2009-06-17 10:37:51Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.parser.BlueprintParser;
import org.jboss.osgi.blueprint.reflect.BlueprintMetadata;
import org.jboss.osgi.spi.util.BundleClassLoader;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.service.blueprint.container.BlueprintContainer;
import org.osgi.service.blueprint.container.NoSuchComponentException;
import org.osgi.service.blueprint.reflect.ComponentMetadata;

/**
 * When a Blueprint extender bundle detects that a Blueprint bundle is
 * ready, it creates a Blueprint Container to manage that Blueprint Bundle.
 * 
 * The Blueprint Container then parses the definitions into metadata objects.
 * All top-level elements in the definitions are ComponentMetadata objects
 * and are registered in the Blueprint Container by their id.
 * 
 * For each of the ComponentMetadata objects, the Blueprint Container has a
 * corresponding component manager. For example, a BeanMetadata object
 * relates to a Bean Manager. There are the following types of managers:
 * 
 * - Bean Managers – Can provide general objects that are properly configured
 * - Service Managers – Can register services
 * - Service Reference Managers – Provide proxies to one or more services. There are two sub-types: ref-list and reference.
 * - Environment Managers – Holding environment values like the Blueprint Bundle object
 * 
 * After creation, all managers are not yet activated. A manager is activated
 * when it has to provide a component instance for the first time. 
 *  
 * @author thomas.diesler@jboss.com
 * @since 17-Jun-2009
 */
public class BlueprintContainerImpl implements BlueprintContainer
{
   public static final String HEADER_BUNDLE_BLUEPRINT = "Bundle-Blueprint";
   public static final String PROPERTY_BLUEPRINT_BUNDLE_SYMBOLIC_NAME = "osgi.blueprint.container.symbolicname";
   public static final String PROPERTY_BLUEPRINT_BUNDLE_VERSION = "osgi.blueprint.container.version";

   private BundleClassLoader classLoader;
   private BlueprintContext context;
   private Bundle bundle;

   private Map<String, AbstractManager> managers = new LinkedHashMap<String, AbstractManager>();

   public BlueprintContainerImpl(BlueprintContext context, Bundle bundle)
   {
      this.context = context;
      this.bundle = bundle;
      
      this.classLoader = BundleClassLoader.createClassLoader(bundle);
   }

   public void initialize()
   {
      // The Blueprint Container parses the definitions into metadata objects.
      BlueprintMetadata bpMetadata = getBlueprintMetadata();

      // For each of the ComponentMetadata objects, the Blueprint Container
      // has a corresponding component manager.
      createManagers(bpMetadata);

      // If the Blueprint Container has successfully activated the eager managers, it
      // will register a Blueprint Container service.
      registerBlueprintContainerService();
   }

   /*
    * When the Blueprint Container must be destroyed because the Blueprint bundle has stopped, there is a failure, or the Blueprint extender is stopped, then the
    * Blueprint Container service is unregistered and all managers are deactivated. This will unregister any services and listeners, which release the component
    * instances. Then all component instances are destroyed in reverse dependency order. That is, a component instance is destroyed when no other component instances
    * depends on it.
    */
   public void shutdown()
   {
      List<AbstractManager> list = new ArrayList<AbstractManager>(managers.values());
      Collections.reverse(list);

      for (AbstractManager manager : list)
         manager.shutdown();
   }
   
   public ClassLoader getClassLoader()
   {
      return classLoader;
   }

   public BundleContext getBundleContext()
   {
      return bundle.getBundleContext();
   }

   public Set<String> getComponentIds()
   {
      return Collections.unmodifiableSet(managers.keySet());
   }

   public ComponentMetadata getComponentMetadata(String name)
   {
      AbstractManager manager = getComponentManager(name);
      return manager.getComponentMetadata();
   }

   @SuppressWarnings("unchecked")
   public <T extends ComponentMetadata> Collection<T> getMetadata(Class<T> type)
   {
      List<T> compMetadata = new ArrayList<T>();
      for (AbstractManager manager : managers.values())
      {
         ComponentMetadata comp = manager.getComponentMetadata();
         if (type.isAssignableFrom(comp.getClass()))
            compMetadata.add((T)comp);
      }
      return Collections.unmodifiableList(compMetadata);
   }
   
   public AbstractManager getComponentManager(String name)
   {
      AbstractManager manager = managers.get(name);
      if (manager == null)
         throw new NoSuchComponentException(name);
      
      return manager;
   }

   public Object getComponentInstance(String id)
   {
      AbstractManager manager = getComponentManager(id);
      return manager.getTargetBean();
   }

   private void registerBlueprintContainerService()
   {
      String symbolicName = bundle.getSymbolicName();
      String version = (String)bundle.getHeaders().get(Constants.BUNDLE_VERSION);

      Properties props = new Properties();
      props.setProperty(PROPERTY_BLUEPRINT_BUNDLE_SYMBOLIC_NAME, symbolicName);
      props.setProperty(PROPERTY_BLUEPRINT_BUNDLE_VERSION, version);
      context.getBundleContext().registerService(BlueprintContainer.class.getName(), this, props);
   }

   private void createManagers(BlueprintMetadata bpMetadata)
   {
      ManagerFactory factory = new ManagerFactory(context, this);
      
      for (ComponentMetadata comp : bpMetadata.getComponents())
      {
         AbstractManager manager = factory.createManager(comp);

         String key = manager.getId();
         if (key == null)
            throw new IllegalStateException("Component id cannot be null for: " + comp);

         // All top level components share a single namespace.
         // That is, it is an error if the same id is used multiple times.
         if (managers.get(key) != null)
            throw new IllegalStateException("Duplicate top-level component: " + key);

         managers.put(key, manager);
      }

      // Invoke install for every manager 
      for (AbstractManager manager : managers.values())
         manager.install();

      // Activated the eager managers
      for (AbstractManager manager : managers.values())
      {
         if (manager.getActivation() == ComponentMetadata.ACTIVATION_EAGER)
            manager.activate();
      }
   }

   private BlueprintMetadata getBlueprintMetadata()
   {
      BlueprintParser parser = context.getBlueprintParser();
      BlueprintMetadata bpMetadata = parser.getBlueprintMetadata(bundle);
      bpMetadata.initialize();
      return bpMetadata;
   }
}