/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.container;

//$Id: ServiceManager.java 90690 2009-06-30 12:09:27Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.reflect.ServiceMetadataImpl;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.blueprint.container.BlueprintContainer;
import org.osgi.service.blueprint.reflect.BeanMetadata;
import org.osgi.service.blueprint.reflect.RefMetadata;
import org.osgi.service.blueprint.reflect.ServiceMetadata;
import org.osgi.service.blueprint.reflect.Target;

/**
 * Service managers maintain the registration of an OSGi service
 * object. Service managers provide a proxied ServiceRegistration object. 
 * 
 * A service manager is enabled if all the mandatory service references in its
 * dependencies are satisfied.
 *  
 * @author thomas.diesler@jboss.com
 * @since 17-Jun-2009
 */
public class ServiceManager extends AbstractManager
{
   private ServiceMetadataImpl serviceMetadata;
   
   private BeanManager anonymousBean;
   private List<ServiceRegistration> serviceRegs = new ArrayList<ServiceRegistration>(); 

   public ServiceManager(BlueprintContext context, BlueprintContainer container, ServiceMetadata compMetadata)
   {
      super(context, container, compMetadata);
      this.serviceMetadata = (ServiceMetadataImpl)compMetadata;

      // Create the anonymous bean manager
      Target target = serviceMetadata.getServiceComponent();
      if (target instanceof BeanMetadata)
      {
         ManagerFactory factory = new ManagerFactory(context, container);
         anonymousBean = (BeanManager)factory.createManager((BeanMetadata)target);         
      }
   }

   @Override
   public void install()
   {
      super.install();
      
      if (anonymousBean != null)
         anonymousBean.install();
   }

   @Override
   public void activate()
   {
      super.activate();
      
      // Get the BeanManager that backs this service
      BeanManager beanManager = anonymousBean;
      if (beanManager == null)
      {
         RefMetadata target = (RefMetadata)serviceMetadata.getServiceComponent();
         BlueprintContainerImpl impl = (BlueprintContainerImpl)container;
         beanManager = (BeanManager)impl.getComponentManager(target.getComponentId());
      }
      
      // Activate the BeanManager that backs this service
      beanManager.activate();
      
      // Register the the target bean as a services
      Object targetBean = beanManager.getTargetBean();
      for (String interf : serviceMetadata.getInterfaces())
      {
         ServiceRegistration sreg = context.getBundleContext().registerService(interf, targetBean, null);
         serviceRegs.add(sreg);
      }
   }

   @Override
   public void shutdown()
   {
      // Unregister the services
      List<ServiceRegistration> list = new ArrayList<ServiceRegistration>(serviceRegs);
      Collections.reverse(list);
      for (ServiceRegistration sreg : list)
         sreg.unregister();
      
      if (anonymousBean != null)
         anonymousBean.shutdown();
      
      super.shutdown();
   }
}