/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.container;

//$Id: AbstractManager.java 93025 2009-08-31 13:10:16Z thomas.diesler@jboss.com $


import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.reflect.ComponentMetadataImpl;
import org.jboss.osgi.spi.service.MicrocontainerService;
import org.osgi.framework.ServiceReference;
import org.osgi.service.blueprint.container.BlueprintContainer;
import org.osgi.service.blueprint.reflect.ComponentMetadata;
import org.osgi.service.log.LogService;

/**
 * For each of the ComponentMetadata objects, the Blueprint Container has a
 * corresponding component manager. For example, a BeanMetadata object
 * relates to a Bean Manager. There are the following types of managers:
 * 
 * • Bean Managers – Can provide general objects that are properly configured
 * • Service Managers – Can register services
 * • Service Reference Managers – Provide proxies to one or more services.
 * • Environment Managers – Holding environment values like the Blueprint Bundle object 
 *   
 * @author thomas.diesler@jboss.com
 * @since 17-Jun-2009
 */
public abstract class AbstractManager
{
   protected LogService log;
   protected BlueprintContext context;
   protected BlueprintContainer container;
   
   private ComponentMetadata component;

   public AbstractManager(BlueprintContext context, BlueprintContainer container, ComponentMetadata component)
   {
      this.log = context.getLog();
      this.context = context;
      this.container = container;
      this.component = component;
   }
   
   public String getId()
   {
      return component.getId();
   }

   public ComponentMetadata getComponentMetadata()
   {
      return component;
   }
   
   public int getActivation()
   {
      return component.getActivation();
   }
   
   public void install()
   {
      log.log(LogService.LOG_DEBUG, "install: " + component);
   }
   
   public void activate()
   {
      log.log(LogService.LOG_DEBUG, "activate: " + component);
   }
   
   public void shutdown()
   {
      log.log(LogService.LOG_DEBUG, "shutdown: " + component);
   }
   
   public Object getTargetBean()
   {
      KernelController controller = getKernelController();
      ComponentMetadataImpl impl = (ComponentMetadataImpl)component;
      ControllerContext ctrlContext = controller.getContext(impl.getKey(), null);
      if (ctrlContext == null)
         throw new IllegalStateException("Cannot obtain controller context for: " + impl.getKey());
      
      return ctrlContext.getTarget();
   }
   
   protected KernelControllerContext installKernelBean(BeanMetaData kernelBean)
   {
      try
      {
         KernelControllerContext ctrlContext = getKernelController().install(kernelBean);
         log.log(LogService.LOG_DEBUG, "installed: " + ctrlContext);
         return ctrlContext;
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Throwable th)
      {
         throw new IllegalStateException("Cannot install kernel bean", th);
      }
   }

   protected void uninstallKernelBean(BeanMetaData kernelBean)
   {
      getKernelController().uninstall(kernelBean.getName());
      log.log(LogService.LOG_DEBUG, "uninstalled: " + kernelBean.getName());
   }

   protected KernelController getKernelController()
   {
      ServiceReference sref = context.getBundleContext().getServiceReference(MicrocontainerService.class.getName());
      if (sref == null)
         throw new IllegalStateException("Cannot obtain MicrocontainerService");
      
      MicrocontainerService service = (MicrocontainerService)context.getBundleContext().getService(sref);
      KernelController controller = service.getRegisteredBean(KernelController.class, MicrocontainerService.BEAN_KERNEL_CONTROLLER);
      if (controller == null)
         throw new IllegalStateException("Cannot obtain KernelController");
      
      return controller;
   }
}