/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.extender;

//$Id: BlueprintProcessor.java 95177 2009-10-20 15:14:31Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.container.BlueprintContainerImpl;
import org.osgi.framework.Bundle;
import org.osgi.framework.Constants;

/**
 * The Blueprint metadata processor
 * 
 * @author thomas.diesler@jboss.com
 * @since 20-Oct-2009
 */
class BlueprintProcessor
{
   private BlueprintContext bpContext;
   private Map<String, BlueprintContainerImpl> containers = new LinkedHashMap<String, BlueprintContainerImpl>();

   public BlueprintProcessor(BlueprintContext bpContext)
   {
      this.bpContext = bpContext;
   }

   public BlueprintContext getBlueprintContext()
   {
      return bpContext;
   }

   @SuppressWarnings("unchecked")
   protected void createContainer(Bundle bundle)
   {
      // If a Bundle-Blueprint manifest header is defined, then this header contains a list of paths. 
      // If this header is not defined, then resources ending in .xml in the bundle’s
      // OSGI-INF/blueprint directory must be used. These are the resources that
      // would be found by calling the Bundle findEntries("OSGI-INF/blueprint", "*.xml", false) method.

      boolean hasBlueprintMetadata = false;

      String descriptorPaths = (String)bundle.getHeaders().get(BlueprintContainerImpl.HEADER_BUNDLE_BLUEPRINT);
      hasBlueprintMetadata = (descriptorPaths != null);

      Enumeration foundEntries = bundle.findEntries("OSGI-INF/blueprint", "*.xml", false);
      hasBlueprintMetadata |= (foundEntries != null);

      if (hasBlueprintMetadata)
      {
         BlueprintContainerImpl container = new BlueprintContainerImpl(bpContext, bundle);
         container.initialize();

         String key = getBundleKey(bundle);
         containers.put(key, container);
      }
   }

   protected boolean hasContainer(Bundle bundle)
   {
      String key = getBundleKey(bundle);
      return containers.containsKey(key);
   }
   
   protected void stopContainer(Bundle bundle)
   {
      String key = getBundleKey(bundle);
      BlueprintContainerImpl container = containers.remove(key);
      if (container != null)
         container.shutdown();
   }

   protected void stopAllContainers()
   {
      Collection<BlueprintContainerImpl> values = containers.values();
      List<BlueprintContainerImpl> list = new ArrayList<BlueprintContainerImpl>(values);
      Collections.reverse(list);

      for (BlueprintContainerImpl container : list)
         container.shutdown();
   }

   private String getBundleKey(Bundle bundle)
   {
      String symbolicName = bundle.getSymbolicName();
      String version = (String)bundle.getHeaders().get(Constants.BUNDLE_VERSION);
      return symbolicName + "-" + version;
   }
}