/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.blueprint.parser;

import static org.jboss.osgi.blueprint.container.BlueprintContainerImpl.HEADER_BUNDLE_BLUEPRINT;

import java.net.URL;
import java.util.Enumeration;

import org.jboss.osgi.blueprint.BlueprintContext;
import org.jboss.osgi.blueprint.parser.xb.TBlueprint;
import org.jboss.osgi.blueprint.reflect.BlueprintMetadata;
import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

//$Id: BlueprintParser.java 97025 2009-11-26 14:35:19Z thomas.diesler@jboss.com $

/**
 * The Blueprint descriptor parser.
 * 
 * @author thomas.diesler@jboss.com
 * @since 13-May-2009
 */
public class BlueprintParser
{
   // Provide Logging
   private Logger log = LoggerFactory.getLogger(BlueprintParser.class);
   
   private BlueprintContext context;
   private JBossXBParser parser;
   
   public BlueprintParser(BlueprintContext context)
   {
      this.context = context;
      this.parser = new SchemaResolverParser(context);
   }
   
   @SuppressWarnings("unchecked")
   public BlueprintMetadata getBlueprintMetadata(Bundle bundle)
   {
      TBlueprint tBlueprint = null;

      // If a Bundle-Blueprint manifest header is defined, then this header contains a list of paths. 
      // If this header is not defined, then resources ending in .xml in the bundle’s
      // OSGI-INF/blueprint directory must be used. These are the resources that
      // would be found by calling the Bundle findEntries("OSGI-INF/blueprint", "*.xml", false) method.

      String descriptorPaths = (String)bundle.getHeaders().get(HEADER_BUNDLE_BLUEPRINT);
      if (descriptorPaths != null)
      {
         log.info(HEADER_BUNDLE_BLUEPRINT + ": " + descriptorPaths + " in bundle: " + bundle);
         String[] descritors = descriptorPaths.split(", ");
         
         for (String descriptor : descritors)
         {
            TBlueprint aux = parse(bundle, descriptor);
            tBlueprint = aux.merge(tBlueprint);
         }
      }

      Enumeration<URL> foundEntries = bundle.findEntries("OSGI-INF/blueprint", "*.xml", false);
      if (foundEntries != null)
      {
         while (foundEntries.hasMoreElements())
         {
            URL nextEntry = foundEntries.nextElement();
            TBlueprint aux = parse(nextEntry);
            tBlueprint = aux.merge(tBlueprint);
         }
      }
      
      if (tBlueprint == null)
         throw new IllegalStateException("Cannot obtain blueprint metadata");
      
      return new BlueprintMetadata(context, tBlueprint);
   }
   
   public TBlueprint parse(Bundle bundle, String descriptor)
   {
      URL descriptorURL = bundle.getResource(descriptor);
      return parse(descriptorURL);
   }

   public TBlueprint parse(URL descriptorURL)
   {
      TBlueprint blueprint = parser.parse(descriptorURL);
      return blueprint;
   }
}