/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky;

// $Id: BridgeFactory.java 90195 2009-06-15 13:10:49Z thomas.diesler@jboss.com $

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.jboss.osgi.husky.internal.BasicBridge;
import org.jboss.osgi.husky.internal.Util;

/**
 * Loads the configured {@link Bridge} instance.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public abstract class BridgeFactory
{
   private static Map<Object, Bridge> bridgeCache = new HashMap<Object, Bridge>(); 
      
   /**
    * Loads the bridge class that is configed through the system property {@link Bridge}.
    * If there is no such property, it returns an instance of {@link BasicBridge}.
    */
   public static Bridge getBridge()
   {
      Properties props = System.getProperties();
      return getBridgeInternal(null, props);
   }

   /**
    * Get the cached instance of the bridge class that is configed through the system property {@link Bridge}.
    * If there is no cached instance it creates a new instance.
    */
   public static Bridge getBridge(Object key)
   {
      Properties props = System.getProperties();
      return getBridgeInternal(key, props);
   }

   /**
    * Loads the bridge class that is configed through the property {@link Bridge}. 
    * 
    * If the loaded bridge has a constructor that takes a {@link Properties} parameter,
    * that constructor is used. Otherwise the bridge will be instanciated with the default constructor
    * 
    * @param props bridge properties
    */
   public static Bridge getBridge(Object key, Properties props)
   {
         String className = props.getProperty(Bridge.class.getName());
         if (className == null)
            throw new IllegalStateException("Cannot find property: " + Bridge.class.getName());

      return getBridgeInternal(key, props);
   }
   
   private static Bridge getBridgeInternal(Object key, Properties props)
   {
      Bridge bridge = bridgeCache.get(key);
      if (bridge == null)
      {
         String className = props.getProperty(Bridge.class.getName(), BasicBridge.class.getName());
         bridge = (Bridge)Util.loadInstance(className, props);
         
         if (key != null)
            bridgeCache.put(key, bridge);
      }
      return bridge;
   }
}
