/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky.internal;

// $Id: AbstractConnector.java 91205 2009-07-14 12:12:00Z thomas.diesler@jboss.com $

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jboss.osgi.husky.Request;
import org.jboss.osgi.husky.Response;
import org.jboss.osgi.husky.runtime.Connector;
import org.jboss.osgi.husky.runtime.PackageListener;

/**
 * An abstract {@link Connector} implementation 
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public abstract class AbstractConnector implements Connector
{
   private List<PackageListener> listeners = new ArrayList<PackageListener>();

   public void addPackageListener(PackageListener listener)
   {
      listeners.add(listener);
   }

   public List<PackageListener> getPackageListeners()
   {
      return Collections.unmodifiableList(listeners);
   }

   public void removePackageListener(PackageListener listener)
   {
      listeners.remove(listener);
   }
   
   public Response process(Request req) throws ClassNotFoundException
   {
      String testClass = req.getClassName();
      for (PackageListener listener : listeners)
      {
         if (listener.match(req))
         {
            return listener.runTests(req);
         }
      }
      throw new IllegalStateException("Cannot find listener to handle: " + testClass + ", we have " + listeners);
   }

   protected InputStream process(InputStream reqStream)
   {
      Request request = null;
      Response response = null;
      try
      {
         // Unmarshall the Request
         ObjectInputStream ois = new ObjectInputStream(reqStream);
         request = (Request)ois.readObject();
         
         // Field the request through the abstract connector
         response = process(request);
      }
      catch (Exception ex)
      {
         response = new BasicResponse();
         BasicFailure failure = new BasicFailure(ex.getMessage(), ex);
         if (request != null)
         {
            failure.setClassName(request.getClassName());
            failure.setMethodName(request.getMethodName());
         }
         response.addFailure(failure);
      }
   
      // Marshall the Response
      try
      {
         ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ObjectOutputStream oos = new ObjectOutputStream(baos);
         oos.writeObject(response);
         oos.close();
         
         return new ByteArrayInputStream(baos.toByteArray());
      }
      catch (IOException ex)
      {
         throw new IllegalStateException("Cannot marshall response", ex);
      }
   }
}