/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky.runtime.junit;

// $Id: JUnitRunner.java 90187 2009-06-15 11:28:40Z thomas.diesler@jboss.com $

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import org.jboss.osgi.husky.Context;
import org.jboss.osgi.husky.Response;
import org.jboss.osgi.husky.annotation.ProvideContext;
import org.jboss.osgi.husky.runtime.Runner;
import org.junit.runner.Result;
import org.junit.runner.notification.RunListener;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.model.InitializationError;

/**
 * An implementation of a {@link Runner} that uses the 
 * JUnit4 Test Runner.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 16-May-2009
 */
public class JUnitRunner implements Runner
{
   /**
    * Run the tests in the given test class.
    */
   public Response runTests(final Context context, final Class<?> testClass)
   {
      // Inject the context in potentially static fields
      try
      {
         injectContext(testClass, null, context);
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception ex)
      {
         throw new IllegalStateException("Cannot initialize test case: " + testClass.getName(), ex);
      }
      
      BlockJUnit4ClassRunner runner;
      try
      {
         runner = new BlockJUnit4ClassRunner(testClass)
         {
            @Override
            protected Object createTest() throws Exception
            {
               Object target = super.createTest();
               injectContext(testClass, target, context);
               return target;
            }
         };
      }
      catch (InitializationError ex)
      {
         throw new IllegalStateException("Cannot initialize test case: " + testClass.getName(), ex);
      }

      Result result = new Result();
      RunListener listener = result.createListener();
      RunNotifier fNotifier = new RunNotifier();
      fNotifier.addFirstListener(listener);
      try
      {
         fNotifier.fireTestRunStarted(runner.getDescription());
         runner.run(fNotifier);
         fNotifier.fireTestRunFinished(result);
      }
      finally
      {
         fNotifier.removeListener(listener);
      }

      Response response = new JUnitResponse(result);
      return response;
   }

   public void injectContext(Class<?> testClass, Object target, Context context) throws Exception
   {
      for (Field field : testClass.getDeclaredFields())
      {
         ProvideContext anProvide = field.getAnnotation(ProvideContext.class);
         if (anProvide != null)
         {
            if ((field.getModifiers() & Modifier.STATIC) != 0 || target != null)
               injectField(field, target, context);
         }
      }
      for (Method method : testClass.getDeclaredMethods())
      {
         ProvideContext anProvide = method.getAnnotation(ProvideContext.class);
         if (anProvide != null)
         {
            if ((method.getModifiers() & Modifier.STATIC) != 0 || target != null)
               injectMethod(method, target, context);
         }
      }
   }

   protected void injectMethod(Method method, Object target, Context value) throws IllegalAccessException, InvocationTargetException
   {
      method.invoke(target, value);
   }

   protected void injectField(Field field, Object target, Context value) throws IllegalAccessException
   {
      field.set(target, value);
   }
}
