/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.husky.runtime.osgi;

// $Id: HuskyExtender.java 91507 2009-07-21 19:30:45Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.jboss.osgi.husky.internal.HuskyLogService;
import org.jboss.osgi.husky.runtime.Connector;
import org.jboss.osgi.husky.runtime.PackageListener;
import org.jboss.osgi.husky.runtime.junit.JUnitPackageListener;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleListener;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.SynchronousBundleListener;
import org.osgi.service.log.LogService;

/**
 * The Husky extender is a {@link BundleListener} that checkes the {@link #HEADER_TEST_PACKAGE}
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 17-May-2009
 */
public class HuskyExtender implements SynchronousBundleListener
{
   /** The manifest header that identifies a bundle as Husky test bundle: 'Test-Package' */
   public static final String HEADER_TEST_PACKAGE = "Test-Package";

   private BundleContext context;
   private LogService log;

   public HuskyExtender(BundleContext context)
   {
      this.log = new HuskyLogService(context);
      this.context = context;
   }

   public void bundleChanged(BundleEvent event)
   {
      if (event.getType() == BundleEvent.STARTED)
      {
         Bundle bundle = event.getBundle();
         List<String> packages = getTestPackages(bundle);
         if (packages != null)
            registerPackageListener(bundle, packages);
      }
      else if (event.getType() == BundleEvent.STOPPED)
      {
         Bundle bundle = event.getBundle();
         List<String> packages = getTestPackages(bundle);
         if (packages != null)
            unregisterPackageListener(bundle, packages);
      }
   }

   private void registerPackageListener(Bundle bundle, List<String> packages)
   {
      ServiceReference[] srefs = null;
      try
      {
         srefs = context.getServiceReferences(Connector.class.getName(), null);
      }
      catch (InvalidSyntaxException ex)
      {
         // cannot get here
      }

      if (srefs != null)
      {
         log.log(LogService.LOG_INFO, "Register test packages " + packages);
         for (ServiceReference sref : srefs)
         {
            Connector connector = (Connector)context.getService(sref);
            JUnitPackageListener listener = new OSGiJUnitPackageListener(bundle, packages);
            connector.addPackageListener(listener);
         }
      }
   }

   private void unregisterPackageListener(Bundle bundle, List<String> packages)
   {
      ServiceReference[] srefs = null;
      try
      {
         srefs = context.getServiceReferences(Connector.class.getName(), null);
      }
      catch (InvalidSyntaxException ex)
      {
         // cannot get here
      }

      if (srefs != null)
      {
         log.log(LogService.LOG_INFO, "Unregister test packages " + packages);
         for (ServiceReference sref : srefs)
         {
            Connector connector = (Connector)context.getService(sref);
            for (PackageListener listener : new ArrayList<PackageListener>(connector.getPackageListeners()))
            {
               List<String> auxNames = listener.getPackageNames();
               if (auxNames.equals(packages))
                  connector.removePackageListener(listener);
            }
         }
      }
   }

   private List<String> getTestPackages(Bundle bundle)
   {
      String testPackage = (String)bundle.getHeaders().get(HEADER_TEST_PACKAGE);
      if (testPackage != null)
         return Arrays.asList(testPackage.split("[,\\s]"));

      return null;
   }
}