/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.testing.internal;

// $Id: RemoteBundle.java 97295 2009-12-02 14:55:02Z thomas.diesler@jboss.com $

import java.util.Dictionary;

import org.jboss.osgi.spi.management.ManagedBundleMBean;
import org.jboss.osgi.spi.util.BundleInfo;
import org.jboss.osgi.spi.util.UnmodifiableDictionary;
import org.jboss.osgi.testing.OSGiBundle;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An implementation of a remote {@link OSGiBundle}
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 25-Sep-2008
 */
public class RemoteBundle extends OSGiBundleImpl
{
   // Provide logging
   private final Logger log = LoggerFactory.getLogger(RemoteBundle.class);
   
   private ManagedBundleMBean bundle;
   private BundleInfo bundleInfo;
   private String location;
   
   private long bundleId;
   private String symbolicName;
   private Dictionary<String, String> headers;
   private Version version;
   
   public RemoteBundle(OSGiRuntimeImpl runtime, ManagedBundleMBean bundle, BundleInfo bundleInfo)
   {
      this(runtime, bundle);
      this.bundleInfo = bundleInfo;
      this.location = bundleInfo.getLocation();
   }
   
   public RemoteBundle(OSGiRuntimeImpl runtime, ManagedBundleMBean bundle)
   {
      super(runtime);
      this.bundle = bundle;
      
      // The getHeaders methods must continue to provide the manifest header
      // information after the bundle enters the UNINSTALLED state.
      
      bundleId = bundle.getBundleId();
      symbolicName = bundle.getSymbolicName();
      headers = bundle.getHeaders();
      
      String versionStr = headers.get(Constants.BUNDLE_VERSION);
      version = Version.parseVersion(versionStr);
   }

   @Override
   public int getState()
   {
      return (isUninstalled() ? Bundle.UNINSTALLED : bundle.getState());
   }

   @Override
   public long getBundleId()
   {
      return bundleId;
   }

   @Override
   public String getSymbolicName()
   {
      return symbolicName;
   }

   @Override
   public Version getVersion()
   {
      return version;
   }

   @Override
   public String getLocation()
   {
      return location;
   }

   @Override
   @SuppressWarnings("unchecked")
   public Dictionary<String, String> getHeaders()
   {
      return new UnmodifiableDictionary(headers);
   }

   @Override
   public String getProperty(String key)
   {
      assertNotUninstalled();
      return bundle.getProperty(key);
   }

   @Override
   public void start() throws BundleException
   {
      assertNotUninstalled();
      bundle.start();
   }

   @Override
   public void stop() throws BundleException
   {
      assertNotUninstalled();
      bundle.stop();
   }

   @Override
   public void uninstall() throws BundleException
   {
      assertNotUninstalled();
      try
      {
         OSGiRuntimeImpl runtimeImpl = (OSGiRuntimeImpl)getRuntime();
         runtimeImpl.undeploy(bundleInfo.getRootURL());
         runtimeImpl.unregisterBundle(this);
         setUninstalled(true);
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception ex)
      {
         log.error("Cannot uninstall: " + getLocation(), ex);
      }
   }
}
