/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.testing.internal;

// $Id: RemoteBundle.java 100478 2010-02-05 12:47:00Z thomas.diesler@jboss.com $

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.util.Dictionary;

import javax.management.InstanceNotFoundException;
import javax.management.ObjectName;

import org.jboss.osgi.spi.management.ManagedBundle;
import org.jboss.osgi.spi.management.ManagedBundleMBean;
import org.jboss.osgi.spi.util.BundleInfo;
import org.jboss.osgi.spi.util.UnmodifiableDictionary;
import org.jboss.osgi.testing.OSGiBundle;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.framework.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An implementation of a remote {@link OSGiBundle}
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 25-Sep-2008
 */
public class RemoteBundle extends OSGiBundleImpl
{
   // Provide logging
   private final Logger log = LoggerFactory.getLogger(RemoteBundle.class);

   private ManagedBundleMBean bundle;
   private String location;

   private long bundleId;
   private String symbolicName;
   private ObjectName objectName;
   private Dictionary<String, String> defaultHeaders;
   private Dictionary<String, String> rawHeaders;
   private Version version;
   boolean uninstalled;

   public RemoteBundle(OSGiRuntimeImpl runtime, ManagedBundleMBean bundle, BundleInfo info)
   {
      this(runtime, bundle);
      this.location = info.getLocation();
   }

   public RemoteBundle(OSGiRuntimeImpl runtime, ManagedBundleMBean bundle)
   {
      super(runtime);
      this.bundle = bundle;

      // The getHeaders methods must continue to provide the manifest header
      // information after the bundle enters the UNINSTALLED state.

      bundleId = bundle.getBundleId();
      symbolicName = bundle.getSymbolicName();
      objectName = bundle.getObjectName();
      location = bundle.getLocation();

      defaultHeaders = bundle.getHeaders(null);
      rawHeaders = bundle.getHeaders("");

      String versionStr = defaultHeaders.get(Constants.BUNDLE_VERSION);
      version = Version.parseVersion(versionStr);
   }

   @Override
   public int getState()
   {
      if (uninstalled == true)
         return Bundle.UNINSTALLED;

      try
      {
         return bundle.getState();
      }
      catch (RuntimeException rte)
      {
         Throwable cause = rte.getCause() != null ? rte.getCause() : rte;
         if (cause instanceof InstanceNotFoundException == false)
            log.warn("Cannot get state for: " + objectName, cause);
      }
      return Bundle.UNINSTALLED;
   }

   @Override
   public long getBundleId()
   {
      return bundleId;
   }

   @Override
   public String getSymbolicName()
   {
      return symbolicName;
   }

   @Override
   public Version getVersion()
   {
      return version;
   }

   @Override
   public String getLocation()
   {
      return location;
   }

   @Override
   @SuppressWarnings("unchecked")
   public Dictionary<String, String> getHeaders()
   {
      return new UnmodifiableDictionary(defaultHeaders);
   }

   @Override
   @SuppressWarnings("unchecked")
   public Dictionary<String, String> getHeaders(String locale)
   {
      if (locale == null)
         return new UnmodifiableDictionary(defaultHeaders);
      else if (locale.length() == 0)
         return new UnmodifiableDictionary(rawHeaders);
      else
         return bundle.getHeaders(locale);
   }

   @Override
   public String getProperty(String key)
   {
      assertNotUninstalled();
      return bundle.getProperty(key);
   }

   @Override
   public URL getEntry(String path)
   {
      assertNotUninstalled();
      return toURL(bundle.getEntry(path), null);
   }

   @Override
   public URL getResource(String name)
   {
      assertNotUninstalled();
      return toURL(bundle.getResource(name), null);
   }

   @Override
   public File getDataFile(String filename)
   {
      assertNotUninstalled();
      return bundle.getDataFile(filename);
   }

   @Override
   public OSGiBundle loadClass(String name) throws ClassNotFoundException
   {
      assertNotUninstalled();
      ObjectName providerBundle = bundle.loadClass(name);
      if (providerBundle == null)
         return null;
      
      String bundleId = providerBundle.getKeyProperty(ManagedBundle.PROPERTY_ID);
      return getRuntime().getBundle(new Long(bundleId));
   }

   @Override
   public void start() throws BundleException
   {
      assertNotUninstalled();
      bundle.start();
   }

   @Override
   public void stop() throws BundleException
   {
      assertNotUninstalled();
      bundle.stop();
   }

   @Override
   public void uninstall() throws BundleException
   {
      assertNotUninstalled();
      try
      {
         OSGiRuntimeImpl runtimeImpl = (OSGiRuntimeImpl)getRuntime();
         runtimeImpl.undeploy(new URL(location));
         runtimeImpl.unregisterBundle(this);
         uninstalled = true;
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception ex)
      {
         log.error("Cannot uninstall: " + getLocation(), ex);
      }
   }

   private URL toURL(String urlstr, URLStreamHandler sh)
   {
      if (urlstr == null)
         return null;

      try
      {
         return sh == null ? new URL(urlstr) : new URL(null, urlstr, sh);
      }
      catch (MalformedURLException ex)
      {
         // In case of the 'bundle' and 'bundleentry' protocol, use a dummy URLStreamHandler
         // Access to remote content via the bundle URL is invalid anyway
         if (sh == null && urlstr.startsWith("bundle"))
         {
            sh = new URLStreamHandler()
            {
               @Override
               protected URLConnection openConnection(URL url) throws IOException
               {
                  return null;
               }
            };
            return toURL(urlstr, sh);
         }
         throw new IllegalArgumentException("Invalid URL: " + urlstr);
      }
   }
}
