/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.testing.internal;

// $Id: RemoteRuntime.java 97666 2009-12-10 05:26:04Z thomas.diesler@jboss.com $

import java.net.URL;
import java.util.HashSet;
import java.util.Set;

import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.osgi.deployment.deployer.DeployerService;
import org.jboss.osgi.deployment.deployer.Deployment;
import org.jboss.osgi.deployment.deployer.DeploymentFactory;
import org.jboss.osgi.spi.management.MBeanProxy;
import org.jboss.osgi.spi.management.MBeanProxyException;
import org.jboss.osgi.spi.management.ManagedBundleMBean;
import org.jboss.osgi.spi.management.ManagedFrameworkMBean;
import org.jboss.osgi.spi.management.ManagedServiceReference;
import org.jboss.osgi.spi.util.BundleInfo;
import org.jboss.osgi.testing.OSGiBundle;
import org.jboss.osgi.testing.OSGiPackageAdmin;
import org.jboss.osgi.testing.OSGiRuntime;
import org.jboss.osgi.testing.OSGiServiceReference;
import org.jboss.osgi.testing.OSGiTestHelper;
import org.osgi.framework.BundleException;

/**
 * A remote implementation of the {@link OSGiRuntime}
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 25-Sep-2008
 */
public class RemoteRuntime extends OSGiRuntimeImpl
{
   private MBeanServerConnection mbeanServer;
   private ManagedFrameworkMBean managedFramework;

   public RemoteRuntime(OSGiTestHelper helper)
   {
      super(helper);
   }

   public OSGiBundle installBundle(String location) throws BundleException
   {
      try
      {
         // Get the bundle info from the location
         BundleInfo info = BundleInfo.createBundleInfo(location);
         Deployment dep = DeploymentFactory.createDeployment(info);
         dep.setAutoStart(false);

         // Deploy the deployemnt through the DeployerService
         DeployerServiceClient deployer = new DeployerServiceClient(this, DeployerService.MBEAN_DEPLOYER_SERVICE);
         deployer.deploy(new Deployment[] { dep });

         String symbolicName = info.getSymbolicName();
         String version = info.getVersion().toString();
         
         // The remote framework uses a BundleTracker that works of an asynchronous 
         // event to register the installed bundle. It is not available immediately.
         int timeout = 50;
         ManagedBundleMBean bundleMBean = getRemoteFramework().getBundle(symbolicName, version);
         while (bundleMBean == null && 0 < timeout--)
         {
            Thread.sleep(200);
            bundleMBean = getRemoteFramework().getBundle(symbolicName, version);
         }
         if (bundleMBean == null)
            throw new IllegalStateException("Cannot find installed bundle: " + symbolicName + "-" + version);

         RemoteBundle bundle = new RemoteBundle(this, bundleMBean, info);
         return registerBundle(bundle.getLocation(), bundle);
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (BundleException ex)
      {
         throw ex;
      }
      catch (Exception ex)
      {
         throw new BundleException("Cannot install: " + location, ex);
      }
   }

   public void deploy(String location) throws Exception
   {
      URL archiveURL = getTestHelper().getTestArchiveURL(location);
      ObjectName oname = getDeployerServiceName(archiveURL);
      DeployerServiceClient deployer = new DeployerServiceClient(this, oname);
      deployer.deploy(archiveURL);
   }

   public void undeploy(String location) throws Exception
   {
      URL archiveURL = getTestHelper().getTestArchiveURL(location);
      ObjectName oname = getDeployerServiceName(archiveURL);
      DeployerServiceClient deployer = new DeployerServiceClient(this, oname);
      deployer.undeploy(archiveURL);
   }

   public OSGiBundle[] getBundles()
   {
      try
      {
         Set<ManagedBundleMBean> remoteBundles = getRemoteFramework().getBundles();
         Set<OSGiBundle> bundles = new HashSet<OSGiBundle>();
         for (ManagedBundleMBean remoteBundle : remoteBundles)
            bundles.add(new RemoteBundle(this, remoteBundle));

         OSGiBundle[] bundleArr = new OSGiBundle[bundles.size()];
         bundles.toArray(bundleArr);
         return bundleArr;
      }
      catch (Exception ex)
      {
         throw new IllegalStateException("Cannot obtain remote bundles", ex);
      }
   }

   public OSGiBundle getBundle(long bundleId)
   {
      ManagedBundleMBean bundle = getRemoteFramework().getBundle(bundleId);
      return bundle != null ? new RemoteBundle(this, bundle) : null;
   }

   public OSGiServiceReference getServiceReference(String clazz)
   {
      ManagedServiceReference manref = getRemoteFramework().getServiceReference(clazz);
      return manref != null ? new RemoteServiceReference(manref) : null;
   }

   public OSGiServiceReference[] getServiceReferences(String clazz, String filter)
   {
      OSGiServiceReference[] srefs = null;

      ManagedServiceReference[] manrefs = getRemoteFramework().getServiceReferences(clazz, filter);
      if (manrefs != null)
      {
         srefs = new OSGiServiceReference[manrefs.length];
         for (int i = 0; i < manrefs.length; i++)
            srefs[i] = new RemoteServiceReference(manrefs[i]);
      }

      return srefs;
   }

   public MBeanServerConnection getMBeanServer()
   {
      if (mbeanServer == null)
      {
         try
         {
            InitialContext iniCtx = getInitialContext();
            mbeanServer = (MBeanServerConnection)iniCtx.lookup("jmx/invoker/RMIAdaptor");
         }
         catch (NamingException ex)
         {
            throw new IllegalStateException("Cannot obtain MBeanServerConnection", ex);
         }
      }
      return mbeanServer;
   }

   public OSGiPackageAdmin getPackageAdmin()
   {
      return new RemotePackageAdmin(this);
   }

   private RemoteFramework getRemoteFramework()
   {
      try
      {
         if (managedFramework == null)
            managedFramework = MBeanProxy.get(ManagedFrameworkMBean.class, ManagedFrameworkMBean.MBEAN_MANAGED_FRAMEWORK, getMBeanServer());
      }
      catch (MBeanProxyException ex)
      {
         throw new RemoteFrameworkException(ex);
      }

      return new RemoteFramework()
      {
         public ManagedBundleMBean getBundle(String name, String version)
         {
            ManagedBundleMBean bundleMBean = null;
            ObjectName oname = managedFramework.getBundle(name, version);
            if (oname != null)
            {
               try
               {
                  bundleMBean = MBeanProxy.get(ManagedBundleMBean.class, oname, getMBeanServer());
               }
               catch (MBeanProxyException ex)
               {
                  throw new RemoteFrameworkException(ex);
               }
            }
            return bundleMBean;
         }

         public ManagedBundleMBean getBundle(long bundleId)
         {
            ManagedBundleMBean bundleMBean = null;
            ObjectName oname = managedFramework.getBundle(bundleId);
            if (oname != null)
            {
               try
               {
                  bundleMBean = MBeanProxy.get(ManagedBundleMBean.class, oname, getMBeanServer());
               }
               catch (MBeanProxyException ex)
               {
                  throw new RemoteFrameworkException(ex);
               }
            }
            return bundleMBean;
         }

         public Set<ManagedBundleMBean> getBundles()
         {
            Set<ManagedBundleMBean> remBundles = new HashSet<ManagedBundleMBean>();
            for (ObjectName bundleOName : managedFramework.getBundles())
            {
               try
               {
                  ManagedBundleMBean remBundle = MBeanProxy.get(ManagedBundleMBean.class, bundleOName, getMBeanServer());
                  remBundles.add(remBundle);
               }
               catch (MBeanProxyException ex)
               {
                  throw new RemoteFrameworkException(ex);
               }
            }
            return remBundles;
         }

         public ManagedServiceReference getServiceReference(String clazz)
         {
            return managedFramework.getServiceReference(clazz);
         }

         public ManagedServiceReference[] getServiceReferences(String clazz, String filter)
         {
            return managedFramework.getServiceReferences(clazz, filter);
         }
      };
   }

   public boolean isRemoteRuntime()
   {
      return true;
   }
   
   /**
    * Return ObjectName of the DeployerService MBean if hte archive is a bundle
    * otherwise if returns the ObjectName of the MainDeployer
    */
   private ObjectName getDeployerServiceName(URL archiveURL) throws MalformedObjectNameException
   {
      ObjectName oname;
      try
      {
         BundleInfo.createBundleInfo(archiveURL);
         oname = DeployerService.MBEAN_DEPLOYER_SERVICE;
      }
      catch (Exception e)
      {
         oname = new ObjectName("jboss.system:service=MainDeployer");
      }
      return oname;
   }
}
