/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossxb.internal;

//$Id: XMLBindingSAXParserFactory.java 97174 2009-11-30 16:09:19Z thomas.diesler@jboss.com $

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.xerces.jaxp.SAXParserFactoryImpl;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.util.xml.XMLParserActivator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

/**
 * A {@link SAXParserFactory}
 * 
 * @author thomas.diesler@jboss.com
 * @since 13-May-2009
 */
public class XMLBindingSAXParserFactory extends SAXParserFactory
{
   // Provide logging
   private static Logger log;

   private static BundleContext context;
   private Map<String, Boolean> features = new HashMap<String, Boolean>();

   public static void setBundleContext(BundleContext context)
   {
      // The log4j initialization routine may use this DocumentBuilderFactory
      // Only create the logger, when we are running in an OSGi context
      log = LoggerFactory.getLogger(XMLBindingSAXParserFactory.class);
      
      XMLBindingSAXParserFactory.context = context;
   }

   @Override
   public boolean getFeature(String name)
   {
      Boolean value = features.get(name);
      return value != null ? value.booleanValue() : false;
   }

   @Override
   public void setFeature(String name, boolean value)
   {
      features.put(name, new Boolean(value));
   }

   /**
    * Creates a new instance of a {@link javax.xml.parsers.SAXParserFactory} using the currently configured parameters.
    * 
    * If the bundle context is available, which would be the case if this bundle was started
    * try using the registered SAXParserFactory service. 
    * 
    * If there is no SAXParserFactory service for the given parameters, use the Apache Xerces SAXParserFactoryImpl.  
    */
   @Override
   public SAXParser newSAXParser() throws ParserConfigurationException, SAXException
   {
      String validating = "(" + XMLParserActivator.PARSER_VALIDATING + "=" + isValidating() + ")";
      String namespaceAware = "(" + XMLParserActivator.PARSER_NAMESPACEAWARE + "=" + isNamespaceAware() + ")";
      String filter = "(&" + validating + namespaceAware + ")";

      SAXParserFactory factory = null;
      
      // First try to obtain the SAXParserFactory service
      if (context != null)
      {
         try
         {
            ServiceReference[] srefs = context.getServiceReferences(SAXParserFactory.class.getName(), filter);
            if (srefs != null)
            {
               if (srefs.length > 1)
                  logDebug("Multiple SAXParserFactory services: " + Arrays.asList(srefs));

               logDebug("Using SAXParserFactory service: " + srefs[0]);
               factory = (SAXParserFactory)context.getService(srefs[0]);
            }
         }
         catch (InvalidSyntaxException ex)
         {
            // ignore
         }
      }

      // Fall back to Apache Xerces SAXParserFactoryImpl
      if (factory == null)
      {
         logDebug("Cannot obtain SAXParserFactory service " + filter + ". Now use " + SAXParserFactoryImpl.class.getName());
         factory = new SAXParserFactoryImpl();
      }

      // Set the features on the factory
      for (Map.Entry<String, Boolean> entry : features.entrySet())
      {
         logDebug("Set feature [" + entry.getKey() + "=" + entry.getValue() + "]");
         factory.setFeature(entry.getKey(), entry.getValue());
      }

      return factory.newSAXParser();
   }

   private void logDebug(String message)
   {
      if (log != null)
         log.debug(message);
   }
}