/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossxb.internal;

//$Id: XMLBindingDocumentBuilderFactory.java 97174 2009-11-30 16:09:19Z thomas.diesler@jboss.com $

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.xerces.jaxp.DocumentBuilderFactoryImpl;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.util.xml.XMLParserActivator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link DocumentBuilderFactory}  
 * 
 * @author thomas.diesler@jboss.com
 * @since 13-May-2009
 */
public class XMLBindingDocumentBuilderFactory extends DocumentBuilderFactory
{
   // Provide logging
   private static Logger log;

   private static BundleContext context;
   private Map<String, Boolean> features = new HashMap<String, Boolean>();
   private Map<String, Object> attributes = new HashMap<String, Object>();

   public static void setBundleContext(BundleContext context)
   {
      // The log4j initialization routine may use this DocumentBuilderFactory
      // Only create the logger, when we are running in an OSGi context
      log = LoggerFactory.getLogger(XMLBindingDocumentBuilderFactory.class);
      
      XMLBindingDocumentBuilderFactory.context = context;
   }

   @Override
   public boolean getFeature(String name)
   {
      Boolean value = features.get(name);
      return value != null ? value.booleanValue() : false;
   }

   @Override
   public void setFeature(String name, boolean value)
   {
      features.put(name, new Boolean(value));
   }

   @Override
   public Object getAttribute(String name)
   {
      return attributes.get(name);
   }

   @Override
   public void setAttribute(String name, Object value)
   {
      attributes.put(name, value);
   }

   /**
    * Creates a new instance of a {@link javax.xml.parsers.DocumentBuilder} using the currently configured parameters.
    * 
    * If the bundle context is available, which would be the case if this bundle was started
    * try using the registered DocumentBuilderFactory service. 
    * 
    * If there is no DocumentBuilderFactory service for the given parameters, use the Apache Xerces DocumentBuilderFactoryImpl.  
    */
   @Override
   public DocumentBuilder newDocumentBuilder() throws ParserConfigurationException
   {
      String validating = "(" + XMLParserActivator.PARSER_VALIDATING + "=" + isValidating() + ")";
      String namespaceAware = "(" + XMLParserActivator.PARSER_NAMESPACEAWARE + "=" + isNamespaceAware() + ")";
      String filter = "(&" + validating + namespaceAware + ")";

      DocumentBuilderFactory factory = null;
      
      // First try to obtain the DocumentBuilderFactory service
      if (context != null)
      {
         try
         {
            ServiceReference[] srefs = context.getServiceReferences(DocumentBuilderFactory.class.getName(), filter);
            if (srefs != null)
            {
               if (srefs.length > 1)
                  logDebug("Multiple DocumentBuilderFactory services: " + Arrays.asList(srefs));

               logDebug("Using DocumentBuilderFactory service: " + srefs[0]);
               factory = (DocumentBuilderFactory)context.getService(srefs[0]);
            }
         }
         catch (InvalidSyntaxException ex)
         {
            // ignore
         }
      }

      // Fall back to Apache Xerces DocumentBuilderFactoryImpl
      if (factory == null)
      {
         logDebug("Cannot obtain DocumentBuilderFactory service " + filter + ". Now use " + DocumentBuilderFactoryImpl.class.getName());
         factory = new DocumentBuilderFactoryImpl();
      }

      // Set the features on the factory
      for (Map.Entry<String, Boolean> entry : features.entrySet())
      {
         logDebug("Set feature [" + entry.getKey() + "=" + entry.getValue() + "]");
         factory.setFeature(entry.getKey(), entry.getValue());
      }

      // Set the attributes on the factory
      for (Map.Entry<String, Object> entry : attributes.entrySet())
      {
         logDebug("Set attribute [" + entry.getKey() + "=" + entry.getValue() + "]");
         factory.setAttribute(entry.getKey(), entry.getValue());
      }

      return factory.newDocumentBuilder();
   }

   private void logDebug(String message)
   {
      if (log != null)
         log.debug(message);
   }
}