/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.deployer;

// $Id: BundleClassLoaderDeployer.java 94643 2009-10-10 21:43:48Z thomas.diesler@jboss.com $

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractSimpleRealDeployer;
import org.jboss.deployers.structure.spi.ClassLoaderFactory;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.structure.spi.DeploymentUnitFilter;
import org.jboss.osgi.deployer.helpers.BundleDeploymentUnitFilter;
import org.jboss.osgi.deployer.helpers.PackageAdminDependencyItem;
import org.jboss.osgi.spi.OSGiConstants;
import org.jboss.osgi.spi.util.BundleClassLoader;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

/**
 * A deployer that attaches a {@link ClassLoaderFactory} that creates a 
 * {@link BundleClassLoader} for the deployed Bundle.  
 *
 * @author Ales.Justin@jboss.org
 * @author Thomas.Diesler@jboss.org
 * @since 03-Feb-2009
 */
public class BundleClassLoaderDeployer extends AbstractSimpleRealDeployer<Bundle>
{
   private DeploymentUnitFilter filter = new BundleDeploymentUnitFilter();
   private BundleContext systemContext;

   public BundleClassLoaderDeployer()
   {
      super(Bundle.class);
      addOutput(ClassLoaderFactory.class);
      setStage(DeploymentStages.POST_PARSE);
   }

   public void setSystemContext(BundleContext systemContext)
   {
      this.systemContext = systemContext;
   }

   public void deploy(DeploymentUnit unit, Bundle bundle) throws DeploymentException
   {
      // create bundle delegate classloader
      ClassLoaderFactory factory = createClassLoaderFactory(bundle);
      unit.addAttachment(ClassLoaderFactory.class, factory);

      // create dependency items
      String propStart = systemContext.getProperty(OSGiConstants.PROPERTY_DEFERRED_START);
      if (Boolean.parseBoolean(propStart != null ? propStart : "true") == false)
         unit.addIDependOn(new PackageAdminDependencyItem(systemContext, bundle));

      // skip annotations scanning, if filter is set
      if (filter != null)
         unit.addAttachment(DeploymentUnitFilter.class, filter);
   }

   /**
    * Create classloader factory.
    *
    * @param bundle the bundle
    * @return new classloader factory
    */
   protected ClassLoaderFactory createClassLoaderFactory(final Bundle bundle)
   {
      return new ClassLoaderFactory()
      {
         public ClassLoader createClassLoader(DeploymentUnit unit) throws Exception
         {
            return BundleClassLoader.createClassLoader(bundle);
         }

         public void removeClassLoader(DeploymentUnit unit) throws Exception
         {
         }
      };
   }

   /**
    * Set deployment filter.
    *
    * @param filter the deployment filter
    */
   public void setFilter(DeploymentUnitFilter filter)
   {
      this.filter = filter;
   }
}