/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.spi.framework;

//$Id: OSGiBootstrap.java 87178 2009-04-12 22:17:09Z thomas.diesler@jboss.com $

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;

/**
 * The OSGiBootstrap provides an {@link OSGiFramework} through a {@link OSGiBootstrapProvider}.
 * 
 * A OSGiBootstrapProvider is discovered in two stages.
 * 
 * <ol>
 * <li>Read the bootstrap provider class name from a system property
 * <li>Read the bootstrap provider class name from a resource file
 * </ol>
 * 
 * In both cases the key is the fully qalified name of the {@link OSGiBootstrapProvider} interface.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public final class OSGiBootstrap
{
   // Hide the ctor
   private OSGiBootstrap()
   {
   }

   /**
    * Get an instance of an OSGiBootstrapProvider.
    */
   public static OSGiBootstrapProvider getBootstrapProvider()
   {
      OSGiBootstrapProvider provider;

      // Get the provider name from the System property
      String providerName = System.getProperty(OSGiBootstrapProvider.class.getName());

      // Get the provider name from the resource file
      if (providerName == null)
      {
         ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
         URL providerURL = ctxLoader.getResource(OSGiBootstrapProvider.class.getName());
         if (providerURL == null)
            throw new IllegalStateException("Cannot find resource: " + OSGiBootstrapProvider.class.getName());

         try
         {
            providerName = new BufferedReader(new InputStreamReader(providerURL.openStream())).readLine();
         }
         catch (Exception e)
         {
            throw new IllegalStateException("Cannot read bootstrap provider name from: " + providerURL);
         }
      }

      // Verify that we have a provider name
      if (providerName == null)
         throw new IllegalStateException("Cannot obtain bootstrap provider");

      // Load the bootstrap provider
      try
      {
         ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
         Class<?> providerClass = ctxLoader.loadClass(providerName);
         provider = (OSGiBootstrapProvider)providerClass.newInstance();
      }
      catch (Exception e)
      {
         throw new IllegalStateException("Cannot load bootstrap provider: " + providerName);
      }

      return provider;
   }
}