/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.spi.junit;

import java.io.File;
import java.io.FileInputStream;
import java.net.URL;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarInputStream;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.deployers.client.spi.DeployerClient;
import org.jboss.osgi.spi.framework.OSGiBootstrapProvider;
import org.jboss.osgi.spi.framework.OSGiFramework;
import org.jboss.osgi.spi.framework.RemoteBundle;
import org.jboss.osgi.spi.framework.RemoteFramework;
import org.jboss.osgi.spi.framework.RemoteFrameworkException;
import org.jboss.osgi.spi.logging.LogEntryCache;
import org.jboss.osgi.spi.logging.RemoteLogReaderService;
import org.jboss.osgi.spi.management.MBeanProxy;
import org.jboss.osgi.spi.management.MBeanProxyException;
import org.jboss.osgi.spi.management.ManagedBundleMBean;
import org.jboss.osgi.spi.management.ManagedFrameworkMBean;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import org.osgi.service.log.LogReaderService;
import org.osgi.util.tracker.ServiceTracker;

/**
 * An integration test helper that deals with test deployment/undeployment, etc.
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 14-Oct-2004
 */
public class IntegrationTestHelper extends OSGiTestHelper
{
   private static MBeanServerConnection server;
   private ManagedFrameworkMBean managedFramework;
   private String integrationTarget;

   public IntegrationTestHelper(OSGiBootstrapProvider bootProvider)
   {
      setBootstrapProvider(bootProvider);
   }

   protected void startRemoteLogEntryTracking(final LogEntryCache logEntryCache) throws Exception
   {
      // Bootstrap the Framework and get the system bundle
      OSGiFramework framework = getBootstrapProvider().getFramework();
      BundleContext sysContext = framework.getSystemBundleContext();
      
      // Track the RemoteLogReaderService to add the LogEntryCache as LogListener
      ServiceTracker tracker = new ServiceTracker(sysContext, RemoteLogReaderService.class.getName(), null)
      {
         @Override
         public Object addingService(ServiceReference sref)
         {
            LogReaderService service = (LogReaderService)super.addingService(sref);
            service.addLogListener(logEntryCache);
            return service;
         }
      };
      tracker.open();

      // Install the RemoteLogReaderService in the local OSGiFramework.
      // The 'org.jboss.osgi.service.remote.log.reader' property must be set to 'true'
      installBundle(sysContext, "bundles/jboss-osgi-remotelog.jar", true);

      // Deploy the RemoteLogListener to the remote OSGiFramework.
      // The 'org.jboss.osgi.service.remote.log.sender' property must be set to 'true'
      deployBundle("bundles/jboss-osgi-remotelog.jar");
   }
   
   protected void stopRemoteLogEntryTracking() throws Exception
   {
      // Undeploy the RemoteLogListener from the remote OSGiFramework.
      undeployBundle("bundles/jboss-osgi-remotelog.jar");
   }
   
   public void deploy(String archive) throws Exception
   {
      URL url = getTestArchiveFile(archive).toURI().toURL();
      deploy(url);
   }

   public void deploy(URL archive) throws Exception
   {
      getDeployer().deploy(archive);
   }

   public void undeploy(String archive) throws Exception
   {
      URL url = getTestArchiveFile(archive).toURI().toURL();
      undeploy(url);
   }

   public void undeploy(URL archive) throws Exception
   {
      getDeployer().undeploy(archive);
   }

   public boolean isTargetJBoss50()
   {
      String target = getIntegrationTarget();
      return target.startsWith("jboss50");
   }

   private String getIntegrationTarget()
   {
      if (integrationTarget == null)
      {
         String jbossVersion;
         try
         {
            ObjectName oname = new ObjectName("jboss.system:type=ServerConfig");
            jbossVersion = (String)getServer().getAttribute(oname, "SpecificationVersion");
         }
         catch (Exception ex)
         {
            throw new IllegalStateException("Cannot obtain jboss version", ex);
         }

         if (jbossVersion.startsWith("5.0"))
            integrationTarget = "jboss50";
         else
            throw new IllegalStateException("Unsupported jboss version: " + jbossVersion);
      }
      return integrationTarget;
   }

   @SuppressWarnings("unchecked")
   public MBeanServerConnection getServer()
   {
      if (server == null)
      {
         Hashtable jndiEnv = null;
         try
         {
            InitialContext iniCtx = new InitialContext();
            jndiEnv = iniCtx.getEnvironment();
            server = (MBeanServerConnection)iniCtx.lookup("jmx/invoker/RMIAdaptor");
         }
         catch (NamingException ex)
         {
            throw new RuntimeException("Cannot obtain MBeanServerConnection using jndi props: " + jndiEnv, ex);
         }
      }
      return server;
   }

   private ArchiveDeployer getDeployer()
   {
      if (isRemoteIntegration())
      {
         return new RemoteArchiveDeployer(getServer());
      }
      else
      {
         DeployerClient deployer = (DeployerClient)getBootstrapProvider().getInstance("MainDeployer");
         return new EmbeddedArchiveDeployer(deployer);
      }
   }

   public RemoteBundle deployBundle(String archive) throws Exception
   {
      File archiveFile = getTestArchiveFile(archive);
      JarInputStream jarInputStream = new JarInputStream(new FileInputStream(archiveFile));
      Attributes attribs = jarInputStream.getManifest().getMainAttributes();
      String symbolicName = attribs.getValue(Constants.BUNDLE_SYMBOLICNAME);
      jarInputStream.close();
      
      if (symbolicName == null)
         throw new IllegalArgumentException("Cannot obtain Bundle-SymbolicName for: " + archive);
      
      deploy(archive);
      
      return getRemoteFramework().getBundle(symbolicName);
   }
   
   public void undeployBundle(String archive) throws Exception
   {
      undeploy(archive);
   }

   public RemoteFramework getRemoteFramework() throws Exception
   {
      if (managedFramework == null)
         managedFramework = MBeanProxy.get(ManagedFrameworkMBean.class, ManagedFrameworkMBean.OBJECT_NAME, getServer());
      
      return new RemoteFramework()
      {
         public RemoteBundle getBundle(String symbolicName)
         {
            ObjectName bundleOName = managedFramework.getBundle(symbolicName);
            try
            {
               return MBeanProxy.get(ManagedBundleMBean.class, bundleOName, getServer());
            }
            catch (MBeanProxyException ex)
            {
               throw new RemoteFrameworkException(ex);
            }
         }

         public Set<RemoteBundle> getBundles()
         {
            Set<RemoteBundle> remBundles = new HashSet<RemoteBundle>();
            for (ObjectName bundleOName : managedFramework.getBundles())
            {
               try
               {
                  RemoteBundle remBundle = MBeanProxy.get(ManagedBundleMBean.class, bundleOName, getServer());
                  remBundles.add(remBundle);
               }
               catch (MBeanProxyException ex)
               {
                  throw new RemoteFrameworkException(ex);
               }
            }
            return remBundles;
         }
      };
   }
   
   public static boolean isRemoteIntegration()
   {
      return System.getProperty("jboss.bind.address") != null;
   }

   public String getServerHost()
   {
      return System.getProperty("jboss.bind.address", "localhost");
   }
}
