/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.spi.testing.internal;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.logging.Logger;
import org.jboss.osgi.spi.capability.Capability;
import org.jboss.osgi.spi.logging.LogEntryCache;
import org.jboss.osgi.spi.testing.OSGiBundle;
import org.jboss.osgi.spi.testing.OSGiRuntime;
import org.jboss.osgi.spi.testing.OSGiServiceReference;
import org.jboss.osgi.spi.testing.OSGiTestHelper;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;
import org.osgi.service.log.LogReaderService;

/**
 * An abstract implementation of the {@link OSGiRuntime}
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 25-Sep-2008
 */
public abstract class OSGiRuntimeImpl implements OSGiRuntime
{
   // Provide logging
   final Logger log = Logger.getLogger(OSGiRuntimeImpl.class);

   private OSGiTestHelper helper;
   private Map<String, OSGiBundle> capBundles = new HashMap<String, OSGiBundle>();
   private List<Capability> capabilities = new ArrayList<Capability>();
   private LogReaderService logReaderService;

   private LogEntryCache logEntryCache;

   public OSGiRuntimeImpl(OSGiTestHelper helper)
   {
      this.helper = helper;
   }

   public OSGiTestHelper getTestHelper()
   {
      return helper;
   }

   protected void setLogReaderService(LogReaderService logReaderService)
   {
      this.logReaderService = logReaderService;
   }

   protected LogReaderService getLogReaderService()
   {
      return logReaderService;
   }

   public void addCapability(Capability capability) throws BundleException
   {
      // Add dependent capabilies
      for (Capability dependency : capability.getDependencies())
         addCapability(dependency);
      
      // Check if the service provided by the capability exists already
      OSGiServiceReference sref = getServiceReference(capability.getServiceName());
      if (sref == null)
      {
         log.debug("Add capability: " + capability);

         for (String location : capability.getBundles())
         {
            String symName = getSymbolicName(location);
            if (capBundles.get(location) == null && getBundle(symName, null) == null)
            {
               OSGiBundle bundle = installBundle(location);
               bundle.start();
               capBundles.put(location, bundle);
            }
            else
            {
               log.debug("Skip bundle: " + location);
            }
         }
         capabilities.add(capability);
      }
      else
      {
         log.debug("Skip capability : " + capability);
      }
   }

   public void removeCapability(Capability capability)
   {
      if (capabilities.remove(capability))
      {
         log.debug("Remove capability : " + capability);

         List<String> bundleLocations = capability.getBundles();
         Collections.reverse(bundleLocations);

         for (String location : bundleLocations)
         {
            OSGiBundle bundle = capBundles.remove(location);
            if (bundle != null)
            {
               try
               {
                  bundle.uninstall();
               }
               catch (BundleException ex)
               {
                  log.error("Cannot uninstall bundle: " + bundle);
               }
            }
         }
      }
      
      List<Capability> dependencies = capability.getDependencies();
      Collections.reverse(dependencies);
      
      // Remove dependent capabilities
      for (Capability dependency : dependencies)
         removeCapability(dependency);
   }

   public void startLogEntryTracking(LogEntryCache logEntryCache)
   {
      this.logEntryCache = logEntryCache;
   }

   public void stopLogEntryTracking()
   {
      if (logReaderService != null && logEntryCache != null)
      {
         logReaderService.removeLogListener(logEntryCache);
         logReaderService = null;
         logEntryCache = null;
      }
   }

   public void shutdown()
   {
      log.debug("Start Shutdown");

      stopLogEntryTracking();

      while (capabilities.size() > 0)
      {
         Capability capability = capabilities.get(0);
         removeCapability(capability);
      }

      log.debug("End Shutdown");
   }

   protected void deploy(String location) throws Exception
   {
      URL archiveURL = getTestHelper().getTestArchiveURL(location);
      invokeDeployerService("deploy", archiveURL);
   }

   protected void undeploy(String location) throws Exception
   {
      URL archiveURL = getTestHelper().getTestArchiveURL(location);
      invokeDeployerService("undeploy", archiveURL);
   }

   private void invokeDeployerService(String method, URL archiveURL) throws Exception
   {
      ObjectName oname = new ObjectName("jboss.osgi:service=DeployerService");
      getMBeanServer().invoke(oname, method, new Object[] { archiveURL }, new String[] { "java.net.URL" });
   }

   public InitialContext getInitialContext() throws NamingException
   {
      return helper.getInitialContext();
   }

   public String getServerHost()
   {
      return helper.getServerHost();
   }

   public OSGiBundle getBundle(String symbolicName, String version)
   {
      OSGiBundle bundle = null;
      for (OSGiBundle aux : getBundles())
      {
         if (aux.getSymbolicName().equals(symbolicName))
         {
            if (version == null || version.equals(aux.getVersion()))
            {
               bundle = aux;
               break;
            }
         }
      }
      return bundle;
   }

   protected String getSymbolicName(String location)
   {
      Manifest manifest;
      try
      {
         File archiveFile = getTestHelper().getTestArchiveFile(location);
         JarFile jarFile = new JarFile(archiveFile);
         manifest = jarFile.getManifest();
         jarFile.close();
      }
      catch (IOException ex)
      {
         throw new IllegalStateException("Cannot get manifest from: " + location);

      }
      
      Attributes attribs = manifest.getMainAttributes();
      String symbolicName = attribs.getValue(Constants.BUNDLE_SYMBOLICNAME);
      if (symbolicName == null)
         throw new IllegalArgumentException("Cannot obtain Bundle-SymbolicName for: " + location);

      return symbolicName;
   }

}
