/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.hibernate.state;


import org.hibernate.Session;
import org.hibernate.Transaction;

import org.jboss.cache.Fqn;
import org.jboss.cache.Modification;
import org.jboss.cache.TreeCache;
import org.jboss.cache.loader.CacheLoader;
import org.jboss.logging.Logger;

import org.jboss.portal.cms.util.HibernateUtil;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;


/*
 * Created on Aug 29, 2006
 *
 * @author <a href="mailto:sohil.shah@jboss.com">Sohil Shah</a>
 */
public class PortalCMSCacheLoader implements CacheLoader
{
   /** Logger instance */
   private static Logger log = Logger.getLogger(PortalCMSCacheLoader.class);

   /**
    *
    */
   private static final String loadWSPProps = "from org.jboss.portal.cms.hibernate.state.WSPProp";
   private static final String loadWSPPropChildren = "select property.propId from org.jboss.portal.cms.hibernate.state.WSPProp property";
   private static final String loadWSPProp = "from org.jboss.portal.cms.hibernate.state.WSPProp property where property.propId=?";
   private static final String loadVersionProps = "from org.jboss.portal.cms.hibernate.state.VersionProp";
   private static final String loadVersionPropChildren = "select property.propId from org.jboss.portal.cms.hibernate.state.VersionProp property";
   private static final String loadVersionProp = "from org.jboss.portal.cms.hibernate.state.VersionProp property where property.propId=?";

   /**
    *
    */
   private static final String loadWSPNodes = "from org.jboss.portal.cms.hibernate.state.WSPNode";
   private static final String loadWSPNodeChildren = "select node.nodeId from org.jboss.portal.cms.hibernate.state.WSPNode node";
   private static final String loadWSPNode = "from org.jboss.portal.cms.hibernate.state.WSPNode node where node.nodeId=?";
   private static final String loadVersionNodes = "from org.jboss.portal.cms.hibernate.state.VersionNode";
   private static final String loadVersionNodeChildren = "select node.nodeId from org.jboss.portal.cms.hibernate.state.VersionNode node";
   private static final String loadVersionNode = "from org.jboss.portal.cms.hibernate.state.VersionNode node where node.nodeId=?";

   /**
    *
    */
   private static final String loadWSPRefs = "from org.jboss.portal.cms.hibernate.state.WSPRefs";
   private static final String loadWSPRefsChildren = "select ref.refId from org.jboss.portal.cms.hibernate.state.WSPRefs ref";
   private static final String loadWSPRef = "from org.jboss.portal.cms.hibernate.state.WSPRefs ref where ref.refId=?";
   private static final String loadVersionRefs = "from org.jboss.portal.cms.hibernate.state.VersionRefs";
   private static final String loadVersionRefsChildren = "select ref.refId from org.jboss.portal.cms.hibernate.state.VersionRefs ref";
   private static final String loadVersionRef = "from org.jboss.portal.cms.hibernate.state.VersionRefs ref where ref.refId=?";


   /**
    *
    */
   public static final String WSP_PROP_NODE = "/wsp_prop";
   public static final String VERSION_PROP_NODE = "/version_prop";
   public static final String WSP_NODE_NODE = "/wsp_node";
   public static final String VERSION_NODE_NODE = "/version_node";
   public static final String WSP_REF_NODE = "/wsp_ref";
   public static final String VERSION_REF_NODE = "/version_ref";

   /** map of propertyId-nodeName map */
   private static Map nodeNames = Collections.synchronizedMap(new HashMap());

   /**
    *
    */
   private TreeCache cache = null;
   private String cmsSessionFactory = null;


   /**
    *
    */
   public void setConfig(Properties properties)
   {
      this.cmsSessionFactory = (String)properties.get("cmsSessionFactory");
   }

   /**
    *
    */
   public void setCache(TreeCache cache)
   {
      this.cache = cache;
   }

   /**
    *
    */
   public Set getChildrenNames(Fqn fqn) throws Exception
   {
      Set children = null;

      if (fqn.toString().equals("/"))
      {
         children = new HashSet();
         children.add(WSP_PROP_NODE.substring(1));
         children.add(VERSION_PROP_NODE.substring(1));
         children.add(WSP_NODE_NODE.substring(1));
         children.add(VERSION_NODE_NODE.substring(1));
      }
      else if (fqn.toString().equals(WSP_PROP_NODE))
      {
         children = this.getChildrenNames(loadWSPPropChildren);
      }
      else if (fqn.toString().equals(VERSION_PROP_NODE))
      {
         children = this.getChildrenNames(loadVersionPropChildren);
      }
      else if (fqn.toString().equals(WSP_NODE_NODE))
      {
         children = this.getChildrenNames(loadWSPNodeChildren);
      }
      else if (fqn.toString().equals(VERSION_NODE_NODE))
      {
         children = this.getChildrenNames(loadVersionNodeChildren);
      }
      else if (fqn.toString().equals(WSP_REF_NODE))
      {
         children = this.getChildrenNames(loadWSPRefsChildren);
      }
      else if (fqn.toString().equals(VERSION_REF_NODE))
      {
         children = this.getChildrenNames(loadVersionRefsChildren);
      }

      if (children != null)
      {
         children = Collections.unmodifiableSet(children);
      }

      return children;
   }

   /**
    *
    */
   public boolean exists(Fqn fqn) throws Exception
   {
      boolean exists = false;

      //node calculation
      String node = fqn.toString();
      String nodeName = "";
      int lastIndex = node.lastIndexOf('/');
      if (lastIndex > 0)
      {
         nodeName = node.substring(lastIndex + 1);
      }
      String query = null;

      if (
         node.equals("/")
         )
      {
         exists = true;
         return exists;
      }
      else if (node.startsWith(WSP_PROP_NODE))
      {
         query = loadWSPProp;
      }
      else if (node.startsWith(VERSION_PROP_NODE))
      {
         query = loadVersionProp;
      }
      else if (node.startsWith(WSP_NODE_NODE))
      {
         query = loadWSPNode;
      }
      else if (node.startsWith(VERSION_NODE_NODE))
      {
         query = loadVersionNode;
      }
      else if (node.startsWith(WSP_REF_NODE))
      {
         query = loadWSPRef;
      }
      else if (node.startsWith(VERSION_REF_NODE))
      {
         query = loadVersionRef;
      }

      if (query != null && query.trim().length() > 0)
      {
         String id = PortalCMSCacheLoader.lookupNodeId(nodeName);
         if (id != null && id.trim().length() > 0)
         {
            Object nodeValue = this.loadNode(query, id);
            if (nodeValue != null)
            {
               exists = true;
            }
         }
      }

      return exists;
   }

   /**
    *
    */
   public Map get(Fqn fqn) throws Exception
   {
      Map map = new HashMap();

      //node calculation
      String node = fqn.toString();
      String nodeName = "";
      int lastIndex = node.lastIndexOf('/');
      if (lastIndex > 0)
      {
         nodeName = node.substring(lastIndex + 1);
      }
      String query = null;

      if (node.startsWith(WSP_PROP_NODE))
      {
         query = loadWSPProp;
      }
      else if (node.startsWith(VERSION_PROP_NODE))
      {
         query = loadVersionProp;
      }
      else if (node.startsWith(WSP_NODE_NODE))
      {
         query = loadWSPNode;
      }
      else if (node.startsWith(VERSION_NODE_NODE))
      {
         query = loadVersionNode;
      }
      else if (node.startsWith(WSP_REF_NODE))
      {
         query = loadWSPRef;
      }
      else if (node.startsWith(VERSION_REF_NODE))
      {
         query = loadVersionRef;
      }

      if (query != null && query.trim().length() > 0)
      {
         String id = PortalCMSCacheLoader.lookupNodeId(nodeName);
         if (id != null && id.trim().length() > 0)
         {
            Object nodeValue = this.loadNode(query, id);
            if (nodeValue != null)
            {
               map.put(id, nodeValue);
            }
         }
      }
      return map;
   }

   /**
    * semantically, if the fqn node does not exist, the entire node needs to be created.. In this cacheLoader's case,
    * the node not existing in the database is not an option. that scenario never happens
    * <p/>
    * returns the oldValue if its found or a new value is created
    */
   public Object put(Fqn fqn, Object key, Object value) throws Exception
   {
      Object oldValue = null;
      String node = fqn.toString();

      if (node.startsWith(WSP_PROP_NODE) && value instanceof WSPProp)
      {
         if (((Base)value).isPersistCacheItem())
         {
            oldValue = this.saveWSPPropNodeEntry((WSPProp)value);
         }
         ((Base)value).resetCacheItemPersistence();
      }
      else if (node.startsWith(VERSION_PROP_NODE) && value instanceof VersionProp)
      {
         if (((Base)value).isPersistCacheItem())
         {
            oldValue = this.saveVersionPropNodeEntry((VersionProp)value);
         }
         ((Base)value).resetCacheItemPersistence();
      }
      else if (node.startsWith(WSP_NODE_NODE) && value instanceof WSPNode)
      {
         if (((Base)value).isPersistCacheItem())
         {
            oldValue = this.saveWSPNodeNodeEntry((WSPNode)value);
         }
         ((Base)value).resetCacheItemPersistence();
      }
      else if (node.startsWith(VERSION_NODE_NODE) && value instanceof VersionNode)
      {
         if (((Base)value).isPersistCacheItem())
         {
            oldValue = this.saveVersionNodeNodeEntry((VersionNode)value);
         }
         ((Base)value).resetCacheItemPersistence();
      }
      else if (node.startsWith(WSP_REF_NODE) && value instanceof WSPRefs)
      {
         if (((Base)value).isPersistCacheItem())
         {
            oldValue = this.saveWSPRefNodeEntry((WSPRefs)value);
         }
         ((Base)value).resetCacheItemPersistence();
      }
      else if (node.startsWith(VERSION_REF_NODE) && value instanceof VersionRefs)
      {
         if (((Base)value).isPersistCacheItem())
         {
            oldValue = this.saveVersionRefNodeEntry((VersionRefs)value);
         }
         ((Base)value).resetCacheItemPersistence();
      }

      return oldValue;
   }

   /**
    *
    */
   public void put(Fqn fqn, Map attributes) throws Exception
   {
      if (attributes != null)
      {
         Set keys = attributes.keySet();
         for (Iterator itr = keys.iterator(); itr.hasNext();)
         {
            Object key = itr.next();
            Object value = attributes.get(key);
            this.put(fqn, key, value);
         }
      }
   }

   /**
    *
    */
   public void put(List modifications) throws Exception
   {
      if (modifications != null)
      {
         for (int i = 0; i < modifications.size(); ++i)
         {
            Modification m = (Modification)modifications.get(i);
            switch (m.getType())
            {
               case Modification.PUT_DATA:
                  put(m.getFqn(), m.getData());
                  break;
               case Modification.PUT_DATA_ERASE:
                  put(m.getFqn(), m.getData());
                  break;
               case Modification.PUT_KEY_VALUE:
                  put(m.getFqn(), m.getKey(), m.getValue());
                  break;
               case Modification.REMOVE_DATA:
                  removeData(m.getFqn());
                  break;
               case Modification.REMOVE_KEY_VALUE:
                  remove(m.getFqn(), m.getKey());
                  break;
               case Modification.REMOVE_NODE:
                  remove(m.getFqn());
                  break;
               default:
                  throw new IllegalStateException("Unexpected modification code: " + m.getType());
            }
         }
      }
   }

   /**
    *
    */
   public Object remove(Fqn fqn, Object key) throws Exception
   {
      Object removedValue = null;
      String node = fqn.toString();
      String id = (String)key;
      String query = null;


      if (node.startsWith(WSP_PROP_NODE))
      {
         query = loadWSPProp;
      }
      else if (node.startsWith(VERSION_PROP_NODE))
      {
         query = loadVersionProp;
      }
      else if (node.startsWith(WSP_NODE_NODE))
      {
         query = loadWSPNode;
      }
      else if (node.startsWith(VERSION_NODE_NODE))
      {
         query = loadVersionNode;
      }
      else if (node.startsWith(WSP_REF_NODE))
      {
         query = loadWSPRef;
      }
      else if (node.startsWith(VERSION_REF_NODE))
      {
         query = loadVersionRef;
      }

      if (query != null && query.trim().length() > 0)
      {
         removedValue = this.delete(query, id);
      }

      return removedValue;
   }

   /**
    *
    */
   public void remove(Fqn fqn) throws Exception
   {
      //just for safety this is not implemented.
      //the cache should not be used for this kind of stuff

      //the admin tool is suited for that
   }

   /**
    *
    */
   public void removeData(Fqn fqn) throws Exception
   {
      //just for safety this is not implemented.
      //the cache should not be used for this kind of stuff

      //the admin tool is suited for that
   }

   //------------cache unit of work transaction-management--------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public void prepare(Object tx, List modifications, boolean one_phase) throws Exception
   {
	  Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
      this.put(modifications);
      session.flush();
   }

   /**
    *
    */
   public void commit(Object tx) throws Exception
   {
   }

   /**
    *
    */
   public void rollback(Object tx)
   {
   }

   //--------------------------------------------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public byte[] loadEntireState() throws Exception
   {
      //loading the entire cms into a byteBuffer and sending it over the network to a remote cache
      //is not feasible for this loader
      //plus, all distributed caches/cache loaders will be loading/saving data to the same database
      //so the content is in sync across all the caches in the cluster
      return null;
   }

   /**
    *
    */
   public void storeEntireState(byte[] arg0) throws Exception
   {
      //no need for this since all the distributed caches/cacheloaders will be loading/saving data to the same
      //database
   }

   //------Service lifecycle implementation---------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public void create() throws Exception
   {
   }

   /**
    *
    */
   public void start() throws Exception
   {
      log.debug("------------------------------------------------------");
      log.debug("PersistenceManager Cache successfully started.....(ClusterName=" + this.cache.getClusterName() + ")");
      log.debug("------------------------------------------------------");
   }

   /**
    *
    */
   public void stop()
   {
      this.cache = null;
   }

   /**
    *
    */
   public void destroy()
   {
   }

   //------internal implementation-----------------------------------------------------------------------------------------------------------------------
   /**
    *
    *
    */
   private Object saveWSPPropNodeEntry(WSPProp wspProp) throws Exception
   {
      Object oldValue = null;

      oldValue = this.loadManagedNode(loadWSPProp, wspProp.getPropId());
      if (oldValue != null)
      {
         Integer id = ((WSPProp)oldValue).getKey();
         wspProp.setKey(id);
      }
      else
      {
         wspProp.setKey(null);
      }
      this.save(wspProp);

      return oldValue;
   }

   /**
    *
    *
    */
   private Object saveVersionPropNodeEntry(VersionProp versionProp) throws Exception
   {
      Object oldValue = null;

      oldValue = this.loadManagedNode(loadVersionProp, versionProp.getPropId());
      if (oldValue != null)
      {
         Integer id = ((VersionProp)oldValue).getKey();
         versionProp.setKey(id);
      }
      else
      {
         versionProp.setKey(null);
      }
      this.save(versionProp);

      return oldValue;
   }

   /**
    *
    *
    */
   private Object saveWSPNodeNodeEntry(WSPNode wspNode) throws Exception
   {
      Object oldValue = null;

      oldValue = this.loadManagedNode(loadWSPNode, wspNode.getNodeId());
      if (oldValue != null)
      {
         Integer id = ((WSPNode)oldValue).getKey();
         wspNode.setKey(id);
      }
      else
      {
         wspNode.setKey(null);
      }
      this.save(wspNode);

      return oldValue;
   }

   /**
    *
    *
    */
   private Object saveVersionNodeNodeEntry(VersionNode versionNode) throws Exception
   {
      Object oldValue = null;

      oldValue = this.loadManagedNode(loadVersionNode, versionNode.getNodeId());
      if (oldValue != null)
      {
         Integer id = ((VersionNode)oldValue).getKey();
         versionNode.setKey(id);
      }
      else
      {
         versionNode.setKey(null);
      }
      this.save(versionNode);

      return oldValue;
   }

   /**
    *
    *
    */
   private Object saveWSPRefNodeEntry(WSPRefs wspRef) throws Exception
   {
      Object oldValue = null;

      oldValue = this.loadManagedNode(loadWSPRef, wspRef.getRefId());
      if (oldValue != null)
      {
         Integer id = ((WSPRefs)oldValue).getKey();
         wspRef.setKey(id);
      }
      else
      {
         wspRef.setKey(null);
      }
      this.save(wspRef);

      return oldValue;
   }

   /**
    *
    *
    */
   private Object saveVersionRefNodeEntry(VersionRefs versionRef) throws Exception
   {
      Object oldValue = null;

      oldValue = this.loadManagedNode(loadVersionRef, versionRef.getRefId());
      if (oldValue != null)
      {
         Integer id = ((VersionRefs)oldValue).getKey();
         versionRef.setKey(id);
      }
      else
      {
         versionRef.setKey(null);
      }
      this.save(versionRef);

      return oldValue;
   }

   //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
   /**
   *
   *
   */
  private Set getChildrenNames(String query) throws Exception
  {
     Set children = null;
     Session session = null;
     session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();

     List rs = session.createQuery(query).list();
     for (Iterator itr = rs.iterator(); itr.hasNext();)
     {
        String id = (String)itr.next();
        if (children == null)
        {
           children = new HashSet();
        }
        children.add(PortalCMSCacheLoader.parseNodeName(id));
     }
     
     return children;
  }

  /**
   *
   *
   */
  private Object loadNode(String query, String id) throws Exception
  {
     Object node = null;
     Session session = null;
     session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();

     //node = session.createQuery(query).setString(0,id).uniqueResult();
     List rs = session.createQuery(query).setString(0, id).list();
     if (rs != null && !rs.isEmpty())
     {
        node = rs.iterator().next();
     }
     
     return node;
  }

  /**
   *
   *
   */
  private Object loadManagedNode(String query, String id) throws Exception
  {
     Object node = null;

     Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
     //node = session.createQuery(query).setString(0,id).uniqueResult();
     List rs = session.createQuery(query).setString(0, id).list();
     if (rs != null && !rs.isEmpty())
     {
        node = rs.iterator().next();
     }

     return node;
  }

  /**
   *
   */
  private void save(Object object) throws Exception
  {
     Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
     if (((Base)object).getKey() != null)
     {
        session.merge(object);
     }
     else
     {
        session.save(object);
     }
  }

  /**
   *
   *
   */
  private Object delete(String loadQuery, String id) throws Exception
  {
     Object removedValue = null;
     removedValue = this.loadManagedNode(loadQuery, id);
     if (removedValue != null)
     {
        Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
        session.delete(removedValue);
     }
     return removedValue;
  }
   //---------------nodeName related methods-------------------------------------------------------------------------------------------------------------
   /**
    *
    *
    */
   public static String parseNodeName(String id)
   {
      String node = (String)PortalCMSCacheLoader.nodeNames.get(id);

      if (node == null)
      {
         node = id.replace('/', '_');
         node = node.replace(':', '_');
         node = node.replace('{', '_');
         node = node.replace('}', '_');
         PortalCMSCacheLoader.nodeNames.put(id, node);
      }

      return node;
   }

   /**
    *
    *
    */
   private static String lookupNodeId(String nodeName)
   {
      String id = null;

      Object[] keys = PortalCMSCacheLoader.nodeNames.keySet().toArray();
      for (int i = 0; i < keys.length; i++)
      {
         String key = (String)keys[i];
         String value = (String)PortalCMSCacheLoader.nodeNames.get(key);
         if (value.equals(nodeName))
         {
            id = key;
            break;
         }
      }

      return id;
   }
}
