/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.interceptors;

import org.jboss.logging.Logger;
import org.jboss.portal.cms.CMSInterceptor;
import org.jboss.portal.cms.impl.CMSCacheException;
import org.jboss.portal.cms.impl.CMSTreeCacheService;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.command.DeleteCommand;
import org.jboss.portal.cms.impl.jcr.command.FileGetCommand;
import org.jboss.portal.cms.impl.jcr.composite.UpdateFileCommand;
import org.jboss.portal.cms.model.File;
import org.jboss.portal.common.invocation.InvocationException;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import java.util.Locale;

/** @author <a href="mailto:theute@jboss.org">Thomas Heute</a> */
public class CacheInterceptor extends CMSInterceptor
{

   private CMSTreeCacheService cache;

   public Logger log = Logger.getLogger(CacheInterceptor.class);


   private void loadCache()
   {
      try
      {
         InitialContext ic = new InitialContext();
         cache = (CMSTreeCacheService)ic.lookup(CMSTreeCacheService.JNDIName);
      }
      catch (NamingException e)
      {
         log.warn("CMSTreeCache service in not available in the JNDI Tree as: " + CMSTreeCacheService.JNDIName);
      }
   }

   protected Object invoke(JCRCommand invocation) throws Exception, InvocationException
   {
      // lazy loading of cache
      if (cache == null)
      {
         loadCache();
      }

      if (invocation instanceof DeleteCommand)
      {
         // delete from cache
         DeleteCommand dc = (DeleteCommand)invocation;
         String path = dc.getPath();
         try
         {
            log.trace(path + " removed from cache");
            cache.remove(path);
         }
         catch (CMSCacheException e)
         {
            e.printStackTrace();
         }
         return invocation.invokeNext();
      }
      else if (invocation instanceof UpdateFileCommand)
      {
         // delete from cache
         UpdateFileCommand ufc = (UpdateFileCommand)invocation;
         try
         {
            String path = ufc.getPath();
            Locale locale = ufc.getLocale();
            log.trace(path + "/" + locale + " removed from cache");
            cache.remove(path, locale);
         }
         catch (CMSCacheException e)
         {
            e.printStackTrace();
         }
         return invocation.invokeNext();
      }
      else if ((invocation instanceof FileGetCommand))
      {
         FileGetCommand fgc = (FileGetCommand)invocation;
         String path = fgc.getPath();
         Locale locale = fgc.getLocale();

         File file = cache.get(path, locale);
         if (file != null)
         {
            log.trace(file.getBasePath() + "/" + locale + " obtained from cache");
            return file;
         }
         else
         {
            File result = (File)invocation.invokeNext();
            if (result != null)
            {
               log.trace(path + "/" + locale + " written to cache");
               cache.put(path, locale, result);
            }
            return result;
         }
      }
      else
      {
         log.trace("Command not trapped by the cache interceptor: " + invocation);
         return invocation.invokeNext();
      }
   }
}
