/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import java.io.File;
import java.io.FileOutputStream;

import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.apache.tools.zip.ZipFile;

import org.jboss.mx.util.MBeanProxy;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.portal.cms.impl.jcr.JCRCMS;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.JCRCommandContext;
import org.jboss.portal.cms.CMS;
import org.jboss.portal.cms.CMSException;


/**
 * Saves an uploaded archive to the repo.
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:sshah@redhat.com">Sohil Shah</a>
 */
public class AsyncStoreArchiveCommand extends JCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 1568453649437987499L;
   private static Logger log = Logger.getLogger(AsyncStoreArchiveCommand.class);
   
   String msRootPath;
   byte[] archiveBytes;
   String msLanguage;
   
   private Object user = null;
   private Object approvePublishWorkflow = null;
   

   /**
    * @param sRootPath
    * @param is
    * @param sLanguage
    */
   public AsyncStoreArchiveCommand(String sRootPath, byte[] archiveBytes, String sLanguage)
   {
      this.msRootPath = sRootPath;
      this.archiveBytes = archiveBytes;
      this.msLanguage = sLanguage;
   }

   /**
    *
    */
   public Object execute()
   {
      try
      {
         //Capture context information to be passed to the Async processor
         this.user = context.getAttribute(JCRCommandContext.scope, "user");
         this.approvePublishWorkflow = context.getAttribute(JCRCommandContext.scope, "approvePublishWorkflow");
         
         //Validate the archive being uploaded
         this.validateArchive();
         
         Thread t = new Thread(new AsyncProcessor());
         t.start();
         
         return null;
      }      
      catch(Exception e)
      {
         if(e instanceof CMSException)
         {
            throw (CMSException)e;
         }
         else
         {
            throw new CMSException(e);
         }
      }
   }
   
   private void validateArchive() throws CMSException
   {
      try
      {
         File tmpFile = this.getZipFile();
         ZipFile zipFile = new ZipFile(tmpFile);
      }
      catch(Exception e)
      {
         throw new CMSException(CMSException.INVALID_ARCHIVE);
      }
   }
   
   private File getZipFile() throws Exception
   {
      File zipFile = null;

      zipFile = File.createTempFile("jbportal_", "_cmsimport.zip");
      FileOutputStream fos = new FileOutputStream(zipFile.getCanonicalPath());
      try
      {
         fos.write(this.archiveBytes);
         fos.close();
      }
      finally
      {
         fos.close();
      }

      return zipFile;
   }
   //--------------------------------------------------------------------------------------------------------------------------------------------------------------
   /**
    * 
    */
   private class AsyncProcessor implements Runnable
   {
      /**
       * 
       */
      public void run()
      {
         try
         {            
            log.info("Async Processing started....................");
            
            //Setup context information to be propagated to the new CMS Service Command call 
            JCRCMS.getUserInfo().set(user);         
            if(approvePublishWorkflow != null)
            {
               JCRCMS.turnOnWorkflow();
            }
            
            CMS cms = this.findCMSService();
            JCRCommand storeArchiveCommand = (JCRCommand)cms.getCommandFactory().createStoreArchiveCommand(msRootPath, archiveBytes, msLanguage);
            cms.execute(storeArchiveCommand);
            
            log.info("Async Processing finished..................");
         }
         catch(Exception e)
         {
            log.error("Asynchronous Processing of archive failed------------------------------------");
            log.error(this, e);
            log.error("-----------------------------------------------------------------------------");
         }
      }
      
      //Find the CMS Service for the Asynchronous processing
      private CMS findCMSService() throws Exception
      {
         MBeanServer mbeanServer = MBeanServerLocator.locateJBoss();
         return (CMS)MBeanProxy.get(CMS.class, new ObjectName("portal:service=CMS"), mbeanServer);
      }
   }   
}