/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.ha;

import java.security.Principal;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.PolicyContextException;

import org.apache.log4j.Logger;
import org.jboss.portal.cms.CMSException;
import org.jboss.portal.cms.Command;
import org.jboss.portal.cms.impl.jcr.JCRCMS;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.JCRCommandContext;
import org.jboss.portal.cms.model.CMSUser;
import org.jboss.portal.identity.User;
import org.jboss.portal.jems.ha.HASingletonInvoker;
import org.jboss.portal.security.impl.jacc.JACCPortalPrincipal;

/**
 * Extend the JCR CMS and make it run as an ha singleton.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @author <a href="mailto:theute@jboss.org">Thomas Heute</a>
 * @version $Revision: 7163 $
 */
public class HAJCRCMS extends JCRCMS implements HASingletonInvoker.Callback
{
   private static Logger log = Logger.getLogger(HAJCRCMS.class);

   /** The invoker. */
   private HASingletonInvoker invoker;

   /** Execute the command. */
   public Object execute(Command cmd) throws CMSException
   {
      if (invoker == null || invoker.isMasterNode())
      {
         // Do it locally
         return super.execute(cmd);
      }
      else
      {
         //Add the user Identity ThreadLocal variable to the Command Context for propagation to the Master Node
         if (this.getUserInfo().get() != null)
         {
            User user = (User)this.getUserInfo().get();
            JCRCommandContext context = this.getCommandContext((JCRCommand)cmd);            
            context.setClusterContextInfo("user", new CMSUser(user.getUserName()));
            
            try
            {
               //Propagate the currently authenticated Subject's roles to the Master Node
               Set<String> roles = this.getCurrentRoles();
               context.setClusterContextInfo("roles", roles);
            }
            catch(PolicyContextException e)
            {
               throw new CMSException(e);
            }           
         }

         //Add the Workflow ThreadLocal variable to the Command Context for propagation to the Master Node
         if (this.turnOffWorkflow.get() != null)
         {
            Boolean workflowStatus = (Boolean)this.turnOffWorkflow.get();
            JCRCommandContext context = this.getCommandContext((JCRCommand)cmd);
            context.setClusterContextInfo("workflowStatus", workflowStatus);
         }
         
         //Add the enableUISecurityFilter indicator to the Command Context for propagation to the Master Node
         if(this.isUISecurityFilterActive())
         {
            JCRCommandContext context = this.getCommandContext((JCRCommand)cmd);
            context.setClusterContextInfo("enableUISecurityFilter", Boolean.TRUE);
         }         

         // Use the proxy to invoke on the singleton
         Object returnValue = null;
         try
         {
            returnValue = invoker.invoke("execute", new Class[]{Command.class}, new Object[]{cmd});
            if(returnValue instanceof CMSException)
            {
               CMSException cmsException = (CMSException)returnValue;
               if(cmsException.toString().indexOf("Access to this resource is denied") != -1 ||
                  cmsException.hasPathFormatFailure() ||
                  cmsException.isServiceUnavailable()
               )
               {
                  throw cmsException;
               }               
               else
               {
                  returnValue = null;
               }
            }
         }
         catch(Exception e)
         {
            if(e instanceof CMSException)
            {
               throw (CMSException)e;
            }
            else
            {
               throw new CMSException(e);
            }
         }
         
         return returnValue;
      }
   }

   /**
    * 
    */
   public void startService() throws Exception
   {
      // Do nothing
   }

   /**
    * 
    */
   public void stopService()
   {
      // Do nothing
   }

   /**
    * 
    */
   public void setInvoker(HASingletonInvoker invoker)
   {
      this.invoker = invoker;
   }

   /**
    * 
    */
   public Object invoke(String methodName, Class[] types, Object[] args) throws CMSException
   {
      Object result = null;
      try
      {
         Command cmd = (Command)args[0];
         result = super.execute(cmd);
      }
      catch(CMSException ce)
      {
         if (ce.toString().indexOf("Access to this resource is denied") != -1 || ce.hasPathFormatFailure())
         {
            result = ce;
         }
      }
      return result;
   }

   /**
    * 
    */
   public String getDisplayName()
   {
      return "HAJCRCMS";
   }

   /**
    * 
    */
   public void startSingleton()
   {
      try
      {
         super.startService();
      }
      catch (Exception e)
      {
         log.error(this, e);
      }
   }

   /**
    * 
    */
   public void stopSingleton()
   {
      super.stopService();
   }
   
   /**
    * 
    * @param command
    * @return
    */
   private JCRCommandContext getCommandContext(JCRCommand command)
   {
      JCRCommandContext context = null;
      
      context = (JCRCommandContext)((JCRCommand)command).getContext();
      if (context == null)
      {
         context = new JCRCommandContext(null, null, null);
         ((JCRCommand)command).setContext(context);
      }
      
      return context;
   }
   
   private Set getCurrentRoles() throws PolicyContextException
   {
      Set<String> roles = new HashSet<String>();

      // Get the current authenticated subject through the JACC contract
      Subject subject = (Subject)PolicyContext.getContext("javax.security.auth.Subject.container");      

      if (subject != null)
      {
         Set tmp = subject.getPrincipals(JACCPortalPrincipal.class);
         JACCPortalPrincipal pp = null;
         for (Iterator i = tmp.iterator(); i.hasNext();)
         {
            pp = (JACCPortalPrincipal)i.next();
            if (pp != null)
            {
               break;
            }
         }
         if (pp == null)
         {
            pp = new JACCPortalPrincipal(subject);
            tmp.add(pp);

            // Lazy create all the permission containers for the given role names
            for (Iterator i = pp.getRoles().iterator(); i.hasNext();)
            {
               Principal role = (Principal)i.next();
               roles.add(role.getName());
            }
         }
      }
      
      return roles;
   }
}
