/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.security;

import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.jboss.mx.util.MBeanProxy;
import org.jboss.portal.cms.util.HibernateUtil;
import org.jboss.portal.cms.impl.jcr.JCRCMS;
import org.jboss.portal.identity.AnonymousRole;
import org.jboss.portal.identity.IdentityContext;
import org.jboss.portal.identity.IdentityServiceController;
import org.jboss.portal.identity.IdentityConfiguration;
import org.jboss.portal.identity.IdentityException;
import org.jboss.portal.identity.MembershipModule;
import org.jboss.portal.identity.Role;
import org.jboss.portal.identity.User;
import org.jboss.portal.identity.UserModule;
import org.jboss.portal.identity.db.HibernateUserModuleImpl;
import org.jboss.portal.jems.as.system.AbstractJBossService;
import org.jboss.portal.jems.hibernate.HibernateProvider;
import org.jboss.portal.jems.hibernate.SessionFactoryBinder;
import org.jboss.portal.security.spi.provider.DomainConfigurator;
import org.jboss.portal.security.spi.provider.PermissionFactory;
import org.jboss.portal.security.spi.provider.PermissionRepository;
import org.jboss.portal.security.spi.provider.SecurityConfigurationException;
import org.jboss.portal.security.impl.jacc.JACCPortalPrincipal;

import javax.management.ObjectName;
import javax.security.auth.Subject;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.PolicyContextException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.security.Principal;

/**
 * Created on : Jan 23, 2007
 *
 * @author Sohil Shah - sohil.shah@jboss.com
 */
public class AuthorizationProviderImpl extends AbstractJBossService implements
   AuthorizationProvider
{
   private IdentityServiceController identityServiceController = null;

   private MembershipModule membershipModule = null;

   private UserModule userModule = null;
      
   private String cmsRootUserName = null;

   private User adminUser = null;
   
   private String cmsSessionFactory = null;

   private String schemaName = null;
   
   /**
    *
    */
   public void startService() throws Exception
   {
      super.startService();
      this.membershipModule = (MembershipModule)identityServiceController
         .getIdentityContext().getObject(
         IdentityContext.TYPE_MEMBERSHIP_MODULE);
      this.userModule = (UserModule)identityServiceController
         .getIdentityContext().getObject(IdentityContext.TYPE_USER_MODULE);

      // We need to access the default schema
      try
      {
    	  HibernateProvider hibernateProvider =  (HibernateProvider)MBeanProxy.get(HibernateProvider.class, new ObjectName("portal:service=Hibernate,type=Instance"), getServer());
    	  if(hibernateProvider!=null) 
    	  {
    		  schemaName = hibernateProvider.getConfig().getProperty("hibernate.default_schema");
    	  }
      }
      catch (Exception e)
      {
    	  // Related to: JBEPP-98
    	  // For the testcase to pass we need to not throw the error.
    	  // This means that we can't create a testcase with default_schema set.
    	  // Will require a rework once fully based on MC.
    	  log.warn("Can't access \"portal:service=Hibernate,type=Instance\" MBean");
      }
   }

   // ------AuthorizationDomain
   // impl-----------------------------------------------------------------
   /**
    *
    */
   public DomainConfigurator getConfigurator()
   {
      return this;
   }

   /**
    *
    */
   public PermissionFactory getPermissionFactory()
   {
      return null;
   }

   /**
    *
    */
   public PermissionRepository getPermissionRepository()
   {
      return null;
   }

   /**
    *
    */
   public String getType()
   {
      return this.getClass().getName();
   }
      
   /**
    * 
    * @return
    */
   public String getCmsRootUserName()
   {
      return cmsRootUserName;
   }

   /**
    * 
    * @param cmsRootUserName
    */
   public void setCmsRootUserName(String cmsRootUserName)
   {
      this.cmsRootUserName = cmsRootUserName;
   }
   
   public String getCmsSessionFactory()
   {
      return cmsSessionFactory;
   }

   public void setCmsSessionFactory(String cmsSessionFactory)
   {
      this.cmsSessionFactory = cmsSessionFactory;
   }
   // ------DomainConfigurator
   // impl------------------------------------------------------------------
   /**
    *
    */
   public void setSecurityBindings(String uri, Set securityBindings)
      throws SecurityConfigurationException
   {
      if (securityBindings != null)
      {
         for (Iterator itr = securityBindings.iterator(); itr.hasNext();)
         {
            Permission securityBinding = (Permission)itr.next();
            if (securityBinding.getRoles() != null
               || securityBinding.getUsers() != null)
            {
               this.store(securityBinding, securityBinding.getRoles(),
                  securityBinding.getUsers());
            }
            else
            {
               this.store(securityBinding);
            }
         }
      }
   }

   /**
    *
    */
   public void removeSecurityBindings(String uri)
      throws SecurityConfigurationException
   {
      Set permissions = this.getSecurityBindings(uri);
      this.remove(permissions);
   }

   /**
    *
    */
   public Set getSecurityBindings(String uri)
   {
      Set permissions = new HashSet();

      if (uri.startsWith("user://"))
      {
         String userId = uri.substring("user://".length());
         Collection cour = this.findPermissionsByUser(userId);
         if (cour != null)
         {
            permissions.addAll(cour);
         }
      }
      else if (uri.startsWith("role://"))
      {
         String roleId = uri.substring("role://".length());
         Collection cour = this.findPermissionsByRole(roleId);
         if (cour != null)
         {
            permissions.addAll(cour);
         }
      }
      else if (uri.startsWith("criteria://"))
      {
         String criteria = uri.substring("criteria://".length());
         int index = criteria.indexOf('/');
         String name = criteria.substring(0, index);
         String value = criteria.substring(index + 1);
         Criteria input = new Criteria(name, value);
         Collection cour = this.findPermissionsByCriteria(input);
         if (cour != null)
         {
            permissions.addAll(cour);
         }
      }

      return permissions;
   }
   
   /**
    * 
    */
   public User getRoot()
   {

      if (adminUser == null)
      {
         Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
         Transaction tx = session.beginTransaction();
         try
         {
            adminUser = this.userModule.findUserByUserName(this.cmsRootUserName);
         }
         catch(Exception e)
         {
        	if(tx != null)
        	{
        		tx.rollback();
        	}
            throw new RuntimeException(e);
         }         
      }
      
      return adminUser;
   }
   
   /**
    * 
    * @return
    */
   public String getDefaultAdminRole()
   {
      try
      {
         String defaultAdminRole = null;
         
         IdentityConfiguration configuration = (IdentityConfiguration)this.identityServiceController.getIdentityContext().
         getObject(IdentityContext.TYPE_IDENTITY_CONFIGURATION);
         defaultAdminRole = configuration.getValue(IdentityConfiguration.ROLE_DEFAULT_ADMIN_ROLE);
         
         return defaultAdminRole;
      }
      catch(IdentityException ie)
      {
         throw new RuntimeException(ie);
      }
   }
   // ---------URIFactory
   // impl-----------------------------------------------------------------------------------------
   /**
    *
    */
   public String getUserURI(String userId)
   {
      return "user://" + userId;
   }

   /**
    * @param roleId
    * @return
    */
   public String getRoleURI(String roleId)
   {
      return "role://" + roleId;
   }

   /**
    * @param name
    * @param value
    * @return
    */
   public String getCriteriaURI(String name, String value)
   {
      return "criteria://" + name + "/" + value;
   }

   /**
    *
    */
   public MembershipModule getMembershipModule()
   {
      return this.membershipModule;
   }

   // --------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public IdentityServiceController getIdentityServiceController()
   {
      return identityServiceController;
   }

   /** @param identityServiceController  */
   public void setIdentityServiceController(
      IdentityServiceController identityServiceController)
   {
      this.identityServiceController = identityServiceController;
   }

   // --------------------------------------------------------------------------------------------------------------
   /**
    * Stores a permission and its associations into peristent storage
    *
    * @param permission - Permission to be persisted
    * @param roles      - Roles associated with this permission
    * @param users      - Individual users associated with this permission
    */
   private void store(Permission permission, Collection roles, Collection users)
   {
      // setup the role association
      if (roles != null)
      {
         Set roleAssoc = new HashSet();
         for (Iterator itr = roles.iterator(); itr.hasNext();)
         {
            Role role = (Role)itr.next();
            PermRoleAssoc cour = new PermRoleAssoc();
            if (!(role instanceof AnonymousRole))
            {
               cour.setRoleId(role.getName());
            }
            else
            {
               cour.setRoleId(AuthorizationManager.Anonymous);
            }
            roleAssoc.add(cour);
         }
         permission.setRoleAssoc(roleAssoc);
      }
      else
      {
         permission.setRoleAssoc(null);
      }

      // setup the user association
      if (users != null)
      {
         Set userAssoc = new HashSet();
         for (Iterator itr = users.iterator(); itr.hasNext();)
         {
            User user = (User)itr.next();
            PermUserAssoc cour = new PermUserAssoc();
            cour.setUserId(user.getUserName());
            userAssoc.add(cour);
         }
         permission.setUserAssoc(userAssoc);
      }
      else
      {
         permission.setUserAssoc(null);
      }

      // persist this into the database
      this.store(permission);
   }

   /**
    * Stores a permission and its associations into peristent storage
    *
    * @param permission - Permission to be persisted
    */
   private void store(Permission permission)
   {
      // persist this into the database
      Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
      session.saveOrUpdate(permission);
   }

   /**
    * Deletes the specified permissions from the database
    *
    * @param permissions
    */
   private void remove(Collection permissions)
   {
      Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
      if (permissions != null)
      {
         for (Iterator itr = permissions.iterator(); itr.hasNext();)
         {
            Permission permission = (Permission)itr.next();
            session.delete(permission);
         }
      }
   }

   /**
    * Returns all permissions associated with the specified user
    *
    * @param user
    * @return
    */
   private Collection findPermissionsByUser(String userId)
   {
      Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
      try
      {
         Collection permissions = new HashSet();

         // this is for db based identity management
         String lookupByUser = "SELECT permission from Permission permission JOIN permission.userAssoc user WHERE user.userId=?";

         // perform lookup by explicitly specified users
         Query userQuery = session.createQuery(lookupByUser);
         userQuery.setString(0, userId);
         userQuery.setCacheable(true);
         permissions.addAll(userQuery.list());
         
         if (this.userModule instanceof HibernateUserModuleImpl)
         {
            String schemaPrefix = (schemaName!=null ? schemaName+"." : "");

            // in case of a database version, use the cache optimized way to
            // lookup the permissions
            String lookupByRole = "SELECT p.* from " + schemaPrefix + "jbp_cms_perm p," + schemaPrefix + "jbp_cms_perm_role r," + schemaPrefix + "jbp_role_membership m," + schemaPrefix + "jbp_roles roles," + schemaPrefix + "jbp_users users WHERE "
               + "p.id=r.cms_perm_id AND "
               + "r.role_id=roles.jbp_name AND "
               + "m.jbp_rid=roles.jbp_rid AND "
               + "m.jbp_uid=users.jbp_uid AND " + "users.jbp_uname=?";

            // perform lookup based on role membership
            Query roleQuery = session.createSQLQuery(lookupByRole).addEntity(
               Permission.class);
            roleQuery.setString(0, userId);
            roleQuery.setCacheable(true);
            permissions.addAll(roleQuery.list());
         }
         else
         {
            

            // in the case of LDAP use the IdentityModule
            // now find permissions for all the roles that this user belongs to
            //Set belongedRoles = this.membershipModule.getRoles(this.userModule
            //   .findUserByUserName(userId));

            Set belongedRoles = getCurrentRoles();
            
            if (belongedRoles != null)
            {
               for (Iterator itr = belongedRoles.iterator(); itr.hasNext();)
               {
                  String role = (String)itr.next();
                  Collection rolePermissions = this.findPermissionsByRole(role);
                  if (rolePermissions != null && !rolePermissions.isEmpty())
                  {
                     permissions.addAll(rolePermissions);
                  }
               }
            }
         }
         return permissions;
      }
      catch (Exception e)
      {
         log.error(this, e);
         throw new RuntimeException(e);
      }      
   }

   /**
    * Returns all permissions associated with the specified role
    *
    * @param role
    * @return
    */
   private Collection findPermissionsByRole(String roleId)
   {
      Collection permissions = new HashSet();

      String lookupByRole = "SELECT permission from Permission permission JOIN permission.roleAssoc role WHERE role.roleId=?";

      Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
      try
      {
         // perform lookup by explicitly specified users
         Query roleQuery = session.createQuery(lookupByRole);
         roleQuery.setString(0, roleId);
         roleQuery.setCacheable(true);
         permissions.addAll(roleQuery.list());
      }
      catch (Exception e)
      {
    	 log.error(this, e);
         throw new RuntimeException(e);
      }
      
      return permissions;
   }

   /**
    * Return all permissions that match the specified criteria
    *
    * @param criteria
    * @return
    */
   private Collection findPermissionsByCriteria(Criteria criteria)
   {
      Collection permissions = new HashSet();

      String lookupByCriteria = "SELECT permission from Permission permission JOIN permission.criteria criteria WHERE criteria.name=? AND criteria.value=?";

      Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
      try
      {
         // perform lookup by explicitly specified users
         Query criteriaQuery = session.createQuery(lookupByCriteria);
         criteriaQuery.setString(0, criteria.getName());
         criteriaQuery.setString(1, criteria.getValue());
         criteriaQuery.setCacheable(true);
         permissions.addAll(criteriaQuery.list());
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }      

      return permissions;
   }

   private Set getCurrentRoles() throws PolicyContextException
   {
      Set<String> roles = new HashSet<String>();

      // Get the current authenticated subject through the JACC contract
      Subject subject = (Subject)PolicyContext.getContext("javax.security.auth.Subject.container");
     
      if (subject != null)
      {
         Set tmp = subject.getPrincipals(JACCPortalPrincipal.class);
         JACCPortalPrincipal pp = null;
         for (Iterator i = tmp.iterator(); i.hasNext();)
         {
            pp = (JACCPortalPrincipal)i.next();
            if (pp != null)
            {
               break;
            }
         }
         if (pp == null)
         {
            pp = new JACCPortalPrincipal(subject);
            tmp.add(pp);

            // Lazy create all the permission containers for the given role names
            for (Iterator i = pp.getRoles().iterator(); i.hasNext();)
            {
               Principal role = (Principal)i.next();
               roles.add(role.getName());
            }
         }
      }
      
      if(roles.isEmpty())
      {
         //Check and see if roles are found propagated via the cluster context
         roles = JCRCMS.getRoles();
      }
      
      return roles;
   }
}
