/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.hibernate;

import org.apache.jackrabbit.core.fs.FileSystem;
import org.apache.jackrabbit.core.fs.FileSystemException;
import org.apache.jackrabbit.core.fs.FileSystemPathUtil;
import org.apache.jackrabbit.core.fs.RandomAccessOutputStream;
import org.apache.jackrabbit.util.TransientFileFactory;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.jboss.logging.Logger;
import org.jboss.portal.cms.util.HibernateUtil;
import org.jboss.portal.common.io.IOTools;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.sql.Blob;
import java.util.Iterator;
import java.util.List;

/**
 * This is a FileSystem store using Hibernate, built on the DBFilesystem class packaged with Apache Jackrabbit that
 * persists file system entries in a database table.
 * <p/>
 * The required schema is created by hibernate using the portal datasource.
 * <p/>
 * Configuration of this store should look like this:
 * <p/>
 * <pre>
 * &lt;FileSystem class="org.jboss.portal.cms.hibernate.HibernateStore"&gt;
 * &lt;param name="datasource" value="java:/PortalDS"/&gt;
 * &lt;param name="isolation" value="2"/&gt;
 * &lt;param name="batch_size" value="0"/&gt;
 * &lt;param name="auto" value="update"/&gt;
 * &lt;param name="provider_class" value="org.hibernate.cache.HashtableCacheProvider"/&gt;
 * &lt;param name="schemaObjectPrefix" value="VersionEntry|CMSEntry|RepositoryEntry"/&gt;
 * &lt;/FileSystem&gt;
 * </pre>
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:theute@jboss.org">Thomas Heute</a>
 */
public class HibernateStore implements FileSystem
{
   private static Logger log = Logger.getLogger(HibernateStore.class);

   private SessionFactory hibernateSessionFactory;

   private String jndiName;

   protected boolean initialized;

   protected String schemaObjectPrefix;

   // initial size of buffer used to serialize objects
   protected static final int INITIAL_BUFFER_SIZE = 8192;

   protected String selectExistStmt;
   protected String selectFolderExistStmt;
   protected String selectFileExistStmt;
   protected String selectChildCountStmt;
   protected String selectDataStmt;
   protected String selectLastModifiedStmt;
   protected String selectLengthStmt;
   protected String deleteFileStmt;
   protected String deleteFolderStmt;
   protected String insertFileStmt;

   /** Default constructor */
   public HibernateStore()
   {
      initialized = false;
   }

   public String getSchemaObjectPrefix()
   {
      return schemaObjectPrefix;
   }

   public void setSchemaObjectPrefix(String schemaObjectPrefix)
   {
      this.schemaObjectPrefix = schemaObjectPrefix;
   }

   public String getJNDIName()
   {
      return jndiName;
   }

   public void setJNDIName(String JNDIName)
   {
      this.jndiName = JNDIName;
   }

   public void init() throws FileSystemException
   {
      if (initialized)
      {
         throw new IllegalStateException("already initialized");
      }

      hibernateSessionFactory = HibernateUtil.getSessionFactory(jndiName);

      selectExistStmt = "select 1 from " + schemaObjectPrefix + " where FSENTRY_PATH = ? " + "and FSENTRY_NAME = ?";

      insertFileStmt = "insert into "
         + schemaObjectPrefix + " (FSENTRY_PATH, FSENTRY_NAME, FSENTRY_DATA, "
         + "FSENTRY_LASTMOD, FSENTRY_LENGTH) "
         + "values (?, ?, ?, ?, ?)";

      selectFileExistStmt = "select 1 from "
         + schemaObjectPrefix + " where FSENTRY_PATH = ? "
         + "and FSENTRY_NAME = ? and FSENTRY_DATA is not null";

      selectFolderExistStmt = "select 1 from "
         + schemaObjectPrefix + " where FSENTRY_PATH = ? "
         + "and FSENTRY_NAME = ? and FSENTRY_DATA is null";

      selectChildCountStmt = "select count(FSENTRY_NAME) from "
         + schemaObjectPrefix + " where FSENTRY_PATH = ?";

      selectDataStmt = "select data from "
         + schemaObjectPrefix + " where FSENTRY_PATH = ? "
         + "and FSENTRY_NAME = ? and FSENTRY_DATA is not null";

      selectLastModifiedStmt = "select FSENTRY_LASTMOD from "
         + schemaObjectPrefix + " where FSENTRY_PATH = ? "
         + "and FSENTRY_NAME = ?";

      selectLengthStmt = "select FSENTRY_LENGTH from "
         + schemaObjectPrefix + " where FSENTRY_PATH = ? "
         + "and FSENTRY_NAME = ? and FSENTRY_DATA is not null";

      initialized = true;
   }

   public void close() throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }
      initialized = false;
   }

   /**
    * Creates a folder and recursively all sub-folders in between
    *
    * @param folderPath the path of the folder to create
    * @throws FileSystemException if the folder cannot be created or if the folder already exists
    */
   public void createFolder(String folderPath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(folderPath);

      if (!exists(folderPath))
      {
         createDeepFolder(folderPath);
      }
      else
      {
         throw new FileSystemException("file system entry already exists: " + folderPath);
      }
   }

   /**
    *
    */
   public void deleteFile(String filePath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(filePath);

      String parentDir = FileSystemPathUtil.getParentDir(filePath);
      String name = FileSystemPathUtil.getName(filePath);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {

         String query = "from " + schemaObjectPrefix + " where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null";

         List result = session.createQuery(query)
            .setString(0, parentDir)
            .setString(1, name)
            .list();

         if (result.size() == 0)
         {
            throw new FileSystemException("no such file: " + filePath);
         }

         if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
         {
            Iterator iter = result.iterator();
            while (iter.hasNext())
            {
               VersionEntry versionEntry = (VersionEntry)iter.next();
               session.delete(versionEntry);
            }

         }
         else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
         {
            Iterator iter = result.iterator();
            while (iter.hasNext())
            {
               CMSEntry cmsEntry = (CMSEntry)iter.next();
               session.delete(cmsEntry);
            }

         }
         else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
         {
            Iterator iter = result.iterator();
            while (iter.hasNext())
            {
               RepositoryEntry repoEntry = (RepositoryEntry)iter.next();
               session.delete(repoEntry);
            }

         }
         session.getTransaction().commit();
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to delete file: " + filePath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public void deleteFolder(String folderPath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(folderPath);

      if (folderPath.equals(FileSystem.SEPARATOR))
      {
         throw new FileSystemException("cannot delete root");
      }

      String parentDir = FileSystemPathUtil.getParentDir(folderPath);
      String name = FileSystemPathUtil.getName(folderPath);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {

         String query = "from " + schemaObjectPrefix + " where FSENTRY_PATH = ? and FSENTRY_NAME = ? " +
            "and FSENTRY_DATA is null or (FSENTRY_PATH = ?) or (FSENTRY_PATH like ?)";

         List result = session.createQuery(query)
            .setString(0, parentDir)
            .setString(1, name)
            .setString(2, folderPath)
            .setString(3, folderPath + FileSystem.SEPARATOR + "%")
            .list();

         if (result.size() == 0)
         {
            throw new FileSystemException("no such folder: " + folderPath);
         }

         if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
         {
            Iterator iter = result.iterator();
            while (iter.hasNext())
            {
               VersionEntry versionEntry = (VersionEntry)iter.next();
               session.delete(versionEntry);

            }

         }
         else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
         {
            Iterator iter = result.iterator();
            while (iter.hasNext())
            {
               CMSEntry cmsEntry = (CMSEntry)iter.next();
               session.delete(cmsEntry);

            }

         }
         else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
         {
            Iterator iter = result.iterator();
            while (iter.hasNext())
            {
               RepositoryEntry repoEntry = (RepositoryEntry)iter.next();
               session.delete(repoEntry);

            }

         }
         session.getTransaction().commit();
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to delete folder: " + folderPath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public boolean exists(String path) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(path);

      String parentDir = FileSystemPathUtil.getParentDir(path);
      String name = FileSystemPathUtil.getName(path);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectExistStmt)
            .setString(0, parentDir)
            .setString(1, name)
            .list();

         Iterator iter = rs.iterator();
         if (iter.hasNext())
         {
            return true;
         }
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to check existence of file system entry: " + path;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
      return false;
   }

   /**
    *
    */
   public boolean isFile(String path) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(path);

      String parentDir = FileSystemPathUtil.getParentDir(path);
      String name = FileSystemPathUtil.getName(path);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectFileExistStmt)
            .setString(0, parentDir)
            .setString(1, name)
            .list();

         Iterator iter = rs.iterator();
         if (iter.hasNext())
         {
            return true;
         }
         
         return false;
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to check existence of file: " + path;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public boolean isFolder(String path) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(path);

      String parentDir = FileSystemPathUtil.getParentDir(path);
      String name = FileSystemPathUtil.getName(path);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectFolderExistStmt)
            .setString(0, parentDir)
            .setString(1, name)
            .list();

         Iterator iter = rs.iterator();
         if (iter.hasNext())
         {
            return true;
         }
         
         return false;
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to check existence of folder: " + path;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public long lastModified(String path) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(path);

      String parentDir = FileSystemPathUtil.getParentDir(path);
      String name = FileSystemPathUtil.getName(path);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectLastModifiedStmt)
            .setString(0, parentDir)
            .setString(1, name)
            .list();

         if (rs.size() == 0)
         {
            throw new FileSystemException("no such file system entry: " + path);
         }
         Iterator iter = rs.iterator();
         return ((Long)iter.next()).longValue();
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to determine lastModified of file system entry: " + path;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public long length(String filePath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(filePath);

      String parentDir = FileSystemPathUtil.getParentDir(filePath);
      String name = FileSystemPathUtil.getName(filePath);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectLengthStmt)
            .setString(0, parentDir)
            .setString(1, name)
            .list();

         if (rs.size() == 0)
         {
            throw new FileSystemException("no such file system entry: " + filePath);
         }
         Iterator iter = rs.iterator();
         return ((Long)iter.next()).longValue();
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to determine length of file: " + filePath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public boolean hasChildren(String path) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(path);

      if (!exists(path))
      {
         throw new FileSystemException("no such file system entry: " + path);
      }

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectChildCountStmt)
            .setString(0, path)
            .list();

         if (rs.size() == 0)
         {
            return false;
         }
         Iterator iter = rs.iterator();
         int count = ((Integer)iter.next()).intValue();
         if (FileSystemPathUtil.denotesRoot(path))
         {
            // ingore file system root entry
            count--;
         }
         return (count > 0);
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to determine child count of file system entry: " + path;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public String[] list(String folderPath) throws FileSystemException
   {
      throw new RuntimeException("Not Yet Implemented");
      /*
      if(!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(folderPath);

      if(!isFolder(folderPath))
      {
         throw new FileSystemException("no such folder: " + folderPath);
      }

      PreparedStatement stmt = selectFileAndFolderNamesStmt;
      synchronized(stmt)
      {
         ResultSet rs = null;
         try
         {
            stmt.setString(1, folderPath);
            stmt.execute();
            rs = stmt.getResultSet();
            ArrayList names = new ArrayList();
            while(rs.next())
            {
               String name = rs.getString(1);
               if(name.length() == 0
                  && FileSystemPathUtil.denotesRoot(folderPath))
               {
                  // this is the file system root entry, skip...
                  continue;
               }
               names.add(name);
            }
            return (String[]) names.toArray(new String[names.size()]);
         }
         catch(Exception e)
         {
            String msg = "failed to list child entries of folder: " + folderPath;
            log.error(msg, e);
            throw new FileSystemException(msg, e);
         }
         finally
         {
//            closeResultSet(rs);
//            resetStatement(stmt);
         }
      }
      */
   }

   /**
    *
    */
   public String[] listFiles(String folderPath) throws FileSystemException
   {
      throw new RuntimeException("Not Yet Implemented");
      /*

      if(!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(folderPath);

      if(!isFolder(folderPath))
      {
         throw new FileSystemException("no such folder: " + folderPath);
      }

      PreparedStatement stmt = selectFileNamesStmt;
      synchronized(stmt)
      {
         ResultSet rs = null;
         try
         {
            stmt.setString(1, folderPath);
            stmt.execute();
            rs = stmt.getResultSet();
            ArrayList names = new ArrayList();
            while(rs.next())
            {
               names.add(rs.getString(1));
            }
            return (String[]) names.toArray(new String[names.size()]);
         }
         catch(Exception e)
         {
            String msg = "failed to list file entries of folder: " + folderPath;
            log.error(msg, e);
            throw new FileSystemException(msg, e);
         }
         finally
         {
//            closeResultSet(rs);
//            resetStatement(stmt);
         }
      }
      */
   }

   /**
    *
    */
   public String[] listFolders(String folderPath) throws FileSystemException
   {
      throw new RuntimeException("Not Yet Implemented");
      /*

      if(!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(folderPath);

      if(!isFolder(folderPath))
      {
         throw new FileSystemException("no such folder: " + folderPath);
      }

      PreparedStatement stmt = selectFolderNamesStmt;
      synchronized(stmt)
      {
         ResultSet rs = null;
         try
         {
            stmt.setString(1, folderPath);
            stmt.execute();
            rs = stmt.getResultSet();
            ArrayList names = new ArrayList();
            while(rs.next())
            {
               String name = rs.getString(1);
               if(name.length() == 0
                  && FileSystemPathUtil.denotesRoot(folderPath))
               {
                  // this is the file system root entry, skip...
                  continue;
               }
               names.add(name);
            }
            return (String[]) names.toArray(new String[names.size()]);
         }
         catch(Exception e)
         {
            String msg = "failed to list folder entries of folder: " + folderPath;
            log.error(msg, e);
            throw new FileSystemException(msg, e);
         }
         finally
         {
//            closeResultSet(rs);
//            resetStatement(stmt);
         }
      }
      */
   }

   /**
    *
    */
   public void touch(String filePath) throws FileSystemException
   {
      throw new RuntimeException("Not Yet Implemented");
      /*

      if(!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(filePath);

      String parentDir = FileSystemPathUtil.getParentDir(filePath);
      String name = FileSystemPathUtil.getName(filePath);

      int count = 0;
      PreparedStatement stmt = updateLastModifiedStmt;
      synchronized(stmt)
      {
         try
         {
            stmt.setLong(1, System.currentTimeMillis());
            stmt.setString(2, parentDir);
            stmt.setString(3, name);
            count = stmt.executeUpdate();
         }
         catch(Exception e)
         {
            String msg = "failed to touch file: " + filePath;
            log.error(msg, e);
            throw new FileSystemException(msg, e);
         }
         finally
         {
//            resetStatement(stmt);
         }
      }

      if(count == 0)
      {
         throw new FileSystemException("no such file: " + filePath);
      }
*/
   }

   /**
    *
    */
   public InputStream getInputStream(String filePath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(filePath);

      String parentDir = FileSystemPathUtil.getParentDir(filePath);
      String name = FileSystemPathUtil.getName(filePath);

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         Query query = session.createQuery(selectDataStmt)
            .setString(0, parentDir)
            .setString(1, name);

         List rs = query.list();
        
         Iterator iter = rs.iterator();
         //byte[] blob = (byte[])iter.next();
         //return new ByteArrayInputStream(blob);
         
         java.sql.Blob blob = (java.sql.Blob)iter.next();                  
         return new ByteArrayInputStream(IOTools.getBytes(blob.getBinaryStream()));
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to retrieve data of file: " + filePath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    *
    */
   public OutputStream getOutputStream(final String filePath)
      throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(filePath);

      final String parentDir = FileSystemPathUtil.getParentDir(filePath);
      final String name = FileSystemPathUtil.getName(filePath);

      if (!isFolder(parentDir))
      {
         throw new FileSystemException("path not found: " + parentDir);
      }

      if (isFolder(filePath))
      {
         throw new FileSystemException("path denotes folder: " + filePath);
      }

      try
      {
         TransientFileFactory fileFactory = TransientFileFactory.getInstance();
         final File tmpFile = fileFactory.createTransientFile("bin", null, null);

         return new FilterOutputStream(new FileOutputStream(tmpFile))
         {
            File f = tmpFile;

            public void close() throws IOException
            {
               super.close();

               InputStream in = null;
               try
               {
                  if (isFile(filePath))
                  {
                     // update
                     Session session = hibernateSessionFactory.getCurrentSession();
                     session.beginTransaction();
                     try
                     {
                        long length = f.length();
                        in = IOTools.safeBufferedWrapper(new FileInputStream(f));
                        if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
                        {
                           Query query = session.createQuery("from VersionEntry where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null");
                           query.setString(0, parentDir);
                           query.setString(1, name);
                           VersionEntry versionEntry = (VersionEntry)query.uniqueResult();
                           if (versionEntry == null)
                           {
                              throw new Exception("No such Entry " + name);
                           }

                           //versionEntry.setData(IOTools.getBytes(in));
                           versionEntry.setData(Hibernate.createBlob(in));
                           
                           versionEntry.setLastmod(System.currentTimeMillis());
                           versionEntry.setLength(length);

                           session.save(versionEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
                        {
                           Query query = session.createQuery("from CMSEntry where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null");
                           query.setString(0, parentDir);
                           query.setString(1, name);
                           CMSEntry cmsEntry = (CMSEntry)query.uniqueResult();
                           if (cmsEntry == null)
                           {
                              throw new Exception("No such Entry " + name);
                           }

                           //cmsEntry.setData(IOTools.getBytes(in));
                           cmsEntry.setData(Hibernate.createBlob(in));
                           
                           cmsEntry.setLastmod(System.currentTimeMillis());
                           cmsEntry.setLength(length);

                           session.save(cmsEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
                        {
                           Query query = session.createQuery("from RepositoryEntry where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null");
                           query.setString(0, parentDir);
                           query.setString(1, name);
                           RepositoryEntry repoEntry = (RepositoryEntry)query.uniqueResult();
                           if (repoEntry == null)
                           {
                              throw new Exception("No such Entry " + name);
                           }

                           //repoEntry.setData(IOTools.getBytes(in));
                           repoEntry.setData(Hibernate.createBlob(in));
                           
                           repoEntry.setLastmod(System.currentTimeMillis());
                           repoEntry.setLength(length);

                           session.save(repoEntry);
                           session.flush();
                        }
                        session.getTransaction().commit();
                     }
                     catch (Exception e)
                     {
                    	session.getTransaction().rollback();
                    	
                        String msg = "failed to update existing file: " + filePath;
                        log.error(msg, e);
                        throw new FileSystemException(msg, e);
                     }                     
                  }
                  else  // create new
                  {
                     Session session = hibernateSessionFactory.getCurrentSession();
                     session.beginTransaction();
                     try
                     {
                        long length = f.length();
                        in = new FileInputStream(f);
                        if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
                        {
                           //VersionEntry versionEntry = new VersionEntry(parentDir, name, IOTools.getBytes(in), System.currentTimeMillis(), length);
                        	VersionEntry versionEntry = new VersionEntry(parentDir, name, Hibernate.createBlob(in), System.currentTimeMillis(), length);
                        	
                           session.save(versionEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
                        {
                           //CMSEntry cmsEntry = new CMSEntry(parentDir, name, IOTools.getBytes(in), System.currentTimeMillis(), length);
                        	CMSEntry cmsEntry = new CMSEntry(parentDir, name, Hibernate.createBlob(in), System.currentTimeMillis(), length);
                        	
                           session.save(cmsEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
                        {
                           //RepositoryEntry repoEntry = new RepositoryEntry(parentDir, name, IOTools.getBytes(in), System.currentTimeMillis(), length);
                        	RepositoryEntry repoEntry = new RepositoryEntry(parentDir, name, Hibernate.createBlob(in), System.currentTimeMillis(), length);
                        	
                           session.save(repoEntry);
                           session.flush();
                        }
                        session.getTransaction().commit();
                     }
                     catch (Exception e)
                     {
                    	session.getTransaction().rollback();
                    	
                        String msg = "failed to create new file: " + filePath;
                        log.error(msg, e);
                        throw new FileSystemException(msg, e);
                     }                     
                  }
               }
               catch (Exception e)
               {
                  throw new IOException(e.getMessage());
               }
               finally
               {
                  IOTools.safeClose(in);
                  // temp file can now safely be removed
                  f.delete();
                  f = null;
               }
            }
         };
      }
      catch (Exception e)
      {
         String msg = "failed to open output strean to file: " + filePath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }
   }

   /**
    *
    */
   public RandomAccessOutputStream getRandomAccessOutputStream(final String filePath)
      throws FileSystemException, UnsupportedOperationException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(filePath);

      final String parentDir = FileSystemPathUtil.getParentDir(filePath);
      final String name = FileSystemPathUtil.getName(filePath);

      if (!isFolder(parentDir))
      {
         throw new FileSystemException("path not found: " + parentDir);
      }

      if (isFolder(filePath))
      {
         throw new FileSystemException("path denotes folder: " + filePath);
      }

      try
      {
         TransientFileFactory fileFactory = TransientFileFactory.getInstance();
         final File tmpFile = fileFactory.createTransientFile("bin", null, null);

// @todo FIXME use java.sql.Blob

         if (isFile(filePath))
         {
            // file entry exists, spool contents to temp file first
            InputStream in = getInputStream(filePath);
            OutputStream out = new FileOutputStream(tmpFile);
            try
            {
               int read;
               byte[] ba = new byte[8192];
               while ((read = in.read(ba, 0, ba.length)) != -1)
               {
                  out.write(ba, 0, read);
               }
            }
            finally
            {
               IOTools.safeClose(out);
//               in.close();
            }
         }

         return new RandomAccessOutputStream()
         {
            File f = tmpFile;
            RandomAccessFile raf = new RandomAccessFile(f, "rw");

            public void close() throws IOException
            {
               raf.close();

               InputStream in = null;
               try
               {
                  if (isFile(filePath))
                  {
                     // update
                     Session session = hibernateSessionFactory.getCurrentSession();
                     session.beginTransaction();
                     try
                     {
                        long length = f.length();
                        in = IOTools.safeBufferedWrapper(new FileInputStream(f));
                        if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
                        {
                           Query query = session.createQuery("from VersionEntry where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null");
                           query.setString(0, parentDir);
                           query.setString(1, name);
                           VersionEntry versionEntry = (VersionEntry)query.uniqueResult();
                           if (versionEntry == null)
                           {
                              throw new Exception("No such Entry " + name);
                           }

                           //versionEntry.setData(IOTools.getBytes(in));
                           versionEntry.setData(Hibernate.createBlob(in));
                           
                           versionEntry.setLastmod(System.currentTimeMillis());
                           versionEntry.setLength(length);

                           session.save(versionEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
                        {
                           Query query = session.createQuery("from CMSEntry where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null");
                           query.setString(0, parentDir);
                           query.setString(1, name);
                           CMSEntry cmsEntry = (CMSEntry)query.uniqueResult();
                           if (cmsEntry == null)
                           {
                              throw new Exception("No such Entry " + name);
                           }

                           //cmsEntry.setData(IOTools.getBytes(in));
                           cmsEntry.setData(Hibernate.createBlob(in));
                           
                           cmsEntry.setLastmod(System.currentTimeMillis());
                           cmsEntry.setLength(length);

                           session.save(cmsEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
                        {
                           Query query = session.createQuery("from CMSEntry where FSENTRY_PATH = ? and FSENTRY_NAME = ? and FSENTRY_DATA is not null");
                           query.setString(0, parentDir);
                           query.setString(1, name);
                           RepositoryEntry repoEntry = (RepositoryEntry)query.uniqueResult();
                           if (repoEntry == null)
                           {
                              throw new Exception("No such Entry " + name);
                           }

                           //repoEntry.setData(IOTools.getBytes(in));
                           repoEntry.setData(Hibernate.createBlob(in));
                           
                           
                           repoEntry.setLastmod(System.currentTimeMillis());
                           repoEntry.setLength(length);

                           session.save(repoEntry);
                           session.flush();
                        }
                        session.getTransaction().commit();
                     }
                     catch (Exception e)
                     {
                    	session.getTransaction().rollback();
                    	
                        String msg = "failed to update existing file: " + filePath;
                        log.error(msg, e);
                        throw new FileSystemException(msg, e);
                     }                     
                  }
                  else
                  {
                     Session session = hibernateSessionFactory.getCurrentSession();
                     session.beginTransaction();
                     try
                     {
                        long length = f.length();
                        in = IOTools.safeBufferedWrapper(new FileInputStream(f));
                        if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
                        {
                           //VersionEntry versionEntry = new VersionEntry(parentDir, name, IOTools.getBytes(in), System.currentTimeMillis(), length);
                        	VersionEntry versionEntry = new VersionEntry(parentDir, name, Hibernate.createBlob(in), System.currentTimeMillis(), length);
                           
                           session.save(versionEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
                        {
                           //CMSEntry cmsEntry = new CMSEntry(parentDir, name, IOTools.getBytes(in), System.currentTimeMillis(), length);
                        	CMSEntry cmsEntry = new CMSEntry(parentDir, name, Hibernate.createBlob(in), System.currentTimeMillis(), length);
                        	
                           session.save(cmsEntry);
                           session.flush();
                        }
                        else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
                        {
                           //RepositoryEntry repoEntry = new RepositoryEntry(parentDir, name, IOTools.getBytes(in), System.currentTimeMillis(), length);
                        	RepositoryEntry repoEntry = new RepositoryEntry(parentDir, name, Hibernate.createBlob(in), System.currentTimeMillis(), length);
                        	
                           session.save(repoEntry);
                           session.flush();
                        }
                        session.getTransaction().commit();
                     }
                     catch (Exception e)
                     {
                    	session.getTransaction().rollback();
                    	
                        String msg = "failed to create new file: " + filePath;
                        log.error(msg, e);
                        throw new FileSystemException(msg, e);
                     }                     
                  }
               }
               catch (Exception e)
               {
                  throw new IOException(e.getMessage());
               }
               finally
               {
                  IOTools.safeClose(in);
                  // temp file can now safely be removed
                  f.delete();
                  f = null;
                  raf = null;
               }
            }

            public void seek(long position) throws IOException
            {
               raf.seek(position);
            }

            public void write(int b) throws IOException
            {
               raf.write(b);
            }

            public void flush() /*throws IOException*/
            {
               // nop
            }

            public void write(byte b[]) throws IOException
            {
               raf.write(b);
            }

            public void write(byte b[], int off, int len) throws IOException
            {
               raf.write(b, off, len);
            }
         };
      }
      catch (Exception e)
      {
         String msg = "failed to open output strean to file: " + filePath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }
   }

   /**
    *
    */
   public void copy(String srcPath, String destPath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(srcPath);
      FileSystemPathUtil.checkFormat(destPath);

      if (isFolder(srcPath))
      {
         // src is a folder
         copyDeepFolder(srcPath, destPath);
      }
      else
      {
         // src is a file
         copyFile(srcPath, destPath);
      }
   }

   /**
    *
    */
   public void move(String srcPath, String destPath) throws FileSystemException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      FileSystemPathUtil.checkFormat(srcPath);
      FileSystemPathUtil.checkFormat(destPath);

// @todo optimize move (use sql update stmts)
      copy(srcPath, destPath);
      if (isFile(srcPath))
      {
         deleteFile(srcPath);
      }
      else
      {
         deleteFolder(srcPath);
      }
   }

//-------------------------------------------------< misc. helper methods >

   /**
    * Checks if the required schema objects exist and creates them if they don't exist yet.
    *
    * @throws Exception if an error occurs
    */
   protected void checkSchema() throws Exception
   {
      throw new RuntimeException("Not Yet Implemented");
      /*
      DatabaseMetaData metaData = con.getMetaData();
      String tableName = schemaObjectPrefix + "FSENTRY";
      if(metaData.storesLowerCaseIdentifiers())
      {
         tableName = tableName.toLowerCase();
      }
      else if(metaData.storesUpperCaseIdentifiers())
      {
         tableName = tableName.toUpperCase();
      }
      ResultSet rs = metaData.getTables(null, null, tableName, null);
      boolean schemaExists;
      try
      {
         schemaExists = rs.next();
      }
      finally
      {
         rs.close();
      }

      if(!schemaExists)
      {
         // read ddl from resources
         InputStream in = getClass().getResourceAsStream(".ddl");
         if(in == null)
         {
            String msg = "Configuration error: unknown schema '" + "'";
            log.debug(msg);
            throw new RepositoryException(msg);
         }
         BufferedReader reader = new BufferedReader(new InputStreamReader(in));
         Statement stmt = con.createStatement();
         try
         {
            String sql = reader.readLine();
            while(sql != null)
            {
               // replace prefix variable
//               sql = Text.replace(sql, SCHEMA_OBJECT_PREFIX_VARIABLE, schemaObjectPrefix);
               // execute sql stmt
               stmt.executeUpdate(sql);
// read next sql stmt
               sql = reader.readLine();
            }
         }
         finally
         {
//            closeStream(in);
//            closeStatement(stmt);
         }
      }
      */
   }

   /**
    * Verifies that the root file system entry exists. If it doesn't exist yet it will be automatically created.
    *
    * @throws Exception if an error occurs
    */
   protected void verifyRootExists() throws Exception
   {
      // check if root file system entry exists
      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(selectFolderExistStmt)
            .setString(1, FileSystem.SEPARATOR)
            .setString(2, "")
            .list();

         Iterator iter = rs.iterator();
         if (iter.hasNext())
         {
            return;
         }
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
         String msg = "failed to check existence of file system root entry";
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      

      // the root entry doesn't exist yet, create it...
      createDeepFolder(FileSystem.SEPARATOR);
   }

   /**
    * Creates the specified files system folder entry, recursively creating any non-existing intermediate folder
    * entries.
    *
    * @param folderPath folder entry to create
    * @throws FileSystemException if an error occurs
    */
   protected void createDeepFolder(String folderPath)
      throws FileSystemException
   {
      String parentDir = FileSystemPathUtil.getParentDir(folderPath);
      String name = FileSystemPathUtil.getName(folderPath);

      if (!FileSystemPathUtil.denotesRoot(folderPath))
      {
         if (!exists(parentDir))
         {
            createDeepFolder(parentDir);
         }
      }

      Session session = hibernateSessionFactory.getCurrentSession();
      session.beginTransaction();
      try
      {
         if (schemaObjectPrefix.equals(HibernateStoreConstants.versionClassName))
         {

            VersionEntry versionEntry = new VersionEntry(parentDir, name, null, System.currentTimeMillis(), 0);
            session.save(versionEntry);
            session.flush();
         }
         else if (schemaObjectPrefix.equals(HibernateStoreConstants.cmsClassName))
         {

            CMSEntry cmsEntry = new CMSEntry(parentDir, name, null, System.currentTimeMillis(), 0);
            session.save(cmsEntry);
         }
         else if (schemaObjectPrefix.equals(HibernateStoreConstants.repositoryClassName))
         {

            RepositoryEntry repoEntry = new RepositoryEntry(parentDir, name, null, System.currentTimeMillis(), 0);
            session.save(repoEntry);
            session.flush();
         }
         session.getTransaction().commit();
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to create folder entry: " + folderPath;
         log.error(msg, e);
         throw new FileSystemException(msg, e);
      }      
   }

   /**
    * Recursively copies the given folder to the given destination.
    *
    * @param srcPath  folder to be copied
    * @param destPath destination path to which the folder is to be copied
    * @throws FileSystemException if an error occurs
    */
   protected void copyDeepFolder(String srcPath, String destPath)
      throws FileSystemException
   {
      throw new RuntimeException("Not Yet Implemented");
/*
      if(!exists(destPath))
      {
         createDeepFolder(destPath);
      }

      String[] names = listFolders(srcPath);

      for(int i = 0; i < names.length; i++)
      {
         String src = (FileSystemPathUtil.denotesRoot(srcPath) ?
                       srcPath + names[i] : srcPath + FileSystem.SEPARATOR + names[i]);
         String dest = (FileSystemPathUtil.denotesRoot(destPath) ?
                        destPath + names[i] : destPath + FileSystem.SEPARATOR + names[i]);
         copyDeepFolder(src, dest);
      }

      PreparedStatement stmt = copyFilesStmt;
      synchronized(stmt)
      {
         try
         {
            stmt.setString(1, srcPath);
            stmt.setString(2, destPath);
            stmt.executeUpdate();
         }
         catch(Exception e)
         {
            String msg = "failed to copy file entries from " + srcPath + " to " + destPath;
            log.error(msg, e);
            throw new FileSystemException(msg, e);
         }
         finally
         {
           resetStatement(stmt);
         }
      }
      */
   }

   /**
    * Copies the given file entry to the given destination path. The parent folder of the destination path will be
    * created if it doesn't exist already. If the destination path refers to an existing file, the file will be
    * overwritten.
    *
    * @param srcPath  file to be copied
    * @param destPath destination path to which the file is to be copied
    * @throws FileSystemException if an error occurs
    */
   protected void copyFile(String srcPath, String destPath)
      throws FileSystemException
   {
      throw new RuntimeException("Not Yet Implemented");
/*
      final String srcParentDir = FileSystemPathUtil.getParentDir(srcPath);
      final String srcName = FileSystemPathUtil.getName(srcPath);

      final String destParentDir = FileSystemPathUtil.getParentDir(destPath);
      final String destName = FileSystemPathUtil.getName(destPath);

      if(!exists(destParentDir))
      {
         createDeepFolder(destParentDir);
      }
      if(isFile(destPath))
      {
         deleteFile(destPath);
      }

      PreparedStatement stmt = copyFileStmt;
      synchronized(stmt)
      {
         try
         {
            stmt.setString(1, destParentDir);
            stmt.setString(2, destName);
            stmt.setString(3, srcParentDir);
            stmt.setString(4, srcName);
            stmt.executeUpdate();
         }
         catch(Exception e)
         {
            String msg = "failed to copy file from " + srcPath + " to " + destPath;
            log.error(msg, e);
            throw new FileSystemException(msg, e);
         }
         finally
         {
            resetStatement(stmt);
         }
      }
      */
   }
}
/*
* Copyright 2004-2005 The Apache Software Foundation or its licensors,
*                     as applicable.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/