/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.hibernate.state;

import org.apache.jackrabbit.core.NodeId;
import org.apache.jackrabbit.core.PropertyId;
import org.apache.jackrabbit.core.fs.FileSystem;
import org.apache.jackrabbit.core.fs.local.LocalFileSystem;
import org.apache.jackrabbit.core.state.ChangeLog;
import org.apache.jackrabbit.core.state.ItemState;
import org.apache.jackrabbit.core.state.ItemStateException;
import org.apache.jackrabbit.core.state.NoSuchItemStateException;
import org.apache.jackrabbit.core.state.NodeReferences;
import org.apache.jackrabbit.core.state.NodeReferencesId;
import org.apache.jackrabbit.core.state.NodeState;
import org.apache.jackrabbit.core.state.PropertyState;
import org.apache.jackrabbit.core.value.BLOBFileValue;
import org.apache.jackrabbit.core.value.InternalValue;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.persistence.PMContext;
import org.apache.jackrabbit.core.persistence.util.FileSystemBLOBStore;
import org.apache.jackrabbit.core.persistence.util.BLOBStore;
import org.apache.jackrabbit.core.persistence.util.Serializer;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;

import org.jboss.cache.CacheException;
import org.jboss.cache.PropertyConfigurator;
import org.jboss.cache.TreeCache;
import org.jboss.cache.TreeCacheMBean;
import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.portal.cms.hibernate.HibernateStoreConstants;
import org.jboss.portal.common.io.IOTools;
import org.jboss.portal.cms.util.HibernateUtil;

import javax.jcr.PropertyType;
import javax.management.MBeanServer;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

/*
 * Created on Aug 28, 2006
 * 
 * @author <a href="mailto:sohil.shah@jboss.com">Sohil Shah</a>
 */
public class JBossCachePersistenceManager implements PersistenceManager
{
   /** Logger instance */
   private static Logger log = Logger
      .getLogger(JBossCachePersistenceManager.class);

   protected static final String SCHEMA_OBJECT_PREFIX_VARIABLE = "${schemaObjectPrefix}";

   protected boolean initialized = false;

   protected String jndiName = null;

   protected String schemaObjectPrefix = null;

   protected boolean externalBLOBs = false;

   // initial size of buffer used to serialize objects
   protected static final int INITIAL_BUFFER_SIZE = 1024;

   // shared statements for BLOB management
   // (if <code>externalBLOBs==false</code>)
   protected String blobSelect = null;

   protected String blobSelectData = null;

   protected String blobSelectExist = null;

   protected String nodeBinValSelect = null;

   /** file system where BLOB data is stored (if <code>externalBLOBs==true</code>) */
   protected FileSystem blobFS = null;

   /** BLOBStore that manages BLOB data in the file system (if <code>externalBLOBs==true</code>) */
   protected BLOBStore blobStore = null;

   /**
    * JackRabbit initalizes multiple instances of the Persistence Manager to perform its operations. A static TreeCache
    * is used to make sure there is only once instance of the cache per VM where the CMS node runs.
    * <p/>
    * One side effect of using a static instance is proper cleanup which is addressed in the close method
    */
   private static TreeCacheMBean pmCache = null;

   private static boolean preloaded = false;

   /**
    *
    *
    */
   public JBossCachePersistenceManager()
   {
      schemaObjectPrefix = "";
      externalBLOBs = true;
      initialized = false;
   }

   // ------------------------------------------------------------------------------------------------------------------------------------------------
   public String getJNDIName()
   {
      return jndiName;
   }

   public void setJNDIName(String JNDIName)
   {
      this.jndiName = JNDIName;
   }

   public String getSchemaObjectPrefix()
   {
      return schemaObjectPrefix;
   }

   public void setSchemaObjectPrefix(String schemaObjectPrefix)
   {
      this.schemaObjectPrefix = schemaObjectPrefix;// .toUpperCase();
   }

   public boolean isExternalBLOBs()
   {
      return externalBLOBs;
   }

   public void setExternalBLOBs(boolean externalBLOBs)
   {
      this.externalBLOBs = externalBLOBs;
   }

   public void setExternalBLOBs(String externalBLOBs)
   {
      this.externalBLOBs = Boolean.valueOf(externalBLOBs).booleanValue();
   }

   /*
    * 
    */
   public void init(PMContext context) throws Exception
   {
      if (initialized)
      {
         throw new IllegalStateException("already initialized");
      }

      nodeBinValSelect = "from " + schemaObjectPrefix
         + "BinVal where BINVAL_ID = ?";

      if (externalBLOBs)
      {
         /**
          * store BLOBs in local file system in a sub directory of the workspace
          * home directory
          */
         LocalFileSystem blobFS = new LocalFileSystem();
         blobFS.setRoot(new File(context.getHomeDir(), "blobs"));
         blobFS.init();
         this.blobFS = blobFS;
         blobStore = new FileSystemBLOBStore(blobFS);
      }
      else
      {
         /**
          * store BLOBs in db
          */
         blobStore = new DbBLOBStore();

         blobSelect = "from " + schemaObjectPrefix
            + "BinVal where BINVAL_ID = ?";
         blobSelectData = "select data from " + schemaObjectPrefix
            + "BinVal where BINVAL_ID = ?";
         blobSelectExist = "select 1 from " + schemaObjectPrefix
            + "BinVal where BINVAL_ID = ?";
      }

      initialized = true;

      try
      {         
         if (pmCache == null)
         {

            /*
             * performing an mbean lookup of the tree cache service. Ideally it
             * would be nice to avoid this and have the service be injected from
             * the sar config. But, a PersistenceManager's lifecycle is managed
             * by JackRabbit and is not easily modeled as a JMX mbean service.
             * 
             * The typical mbean lifecycle doesn't map here unless the PM serves
             * as a wrapper/proxy to an mbean service, but not sure if the extra
             * layer is necessary.
             * 
             * Also, see the close method to see how the PM cache service is
             * cleaned up.
             */
            try
            {
               MBeanServer server = MBeanServerLocator.locateJBoss();
               pmCache = (TreeCacheMBean)MBeanProxyExt.create(
                  TreeCacheMBean.class, "cms.pm.cache:service=TreeCache",
                  server);
            }
            catch (Exception e)
            {
            }

            // try to load from specified configuration if any
            // if nothing found in the environment...chances are running in a
            // non-managed environment
            if (pmCache == null)
            {
               InputStream is = null;
               try
               {
                  pmCache = new TreeCache();

                  // configure the cache
                  PropertyConfigurator configurator = new PropertyConfigurator();
                  is = JBossCachePersistenceManager.class.getClassLoader()
                     .getResourceAsStream("pm-cache.xml");
                  configurator.configure(pmCache, is);

                  pmCache.createService();
                  pmCache.startService();
               }
               finally
               {
                  if (is != null)
                  {
                     is.close();
                  }
               }
            }
         }

         // pre-load the cache with property nodes
         if (!JBossCachePersistenceManager.preloaded)
         {
            JBossCachePersistenceManager.loadProperties(this.jndiName);
            log.info("-------------------------------------------------");
            log.info("JBossCachePersistenceManager is fully loaded.....");
            log.info("-------------------------------------------------");
         }
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    *
    *
    */
   private static synchronized void loadProperties(String hibernateJndiName) throws Exception
   {
      if (JBossCachePersistenceManager.preloaded)
      {
         return;
      }

      log
         .info("Pre-loading the PersistenceManager Cache in the background (started).......");

      CacheLoader wspProp = new CacheLoader(CacheLoader.wspProp,
         PortalCMSCacheLoader.WSP_PROP_NODE, hibernateJndiName);
      Thread wspPropLoader = new Thread(wspProp);
      CacheLoader versionProp = new CacheLoader(CacheLoader.versionProp,
         PortalCMSCacheLoader.VERSION_PROP_NODE, hibernateJndiName);
      Thread versionPropLoader = new Thread(versionProp);
      CacheLoader wspNode = new CacheLoader(CacheLoader.wspNode,
         PortalCMSCacheLoader.WSP_NODE_NODE, hibernateJndiName);
      Thread wspNodeLoader = new Thread(wspNode);
      CacheLoader versionNode = new CacheLoader(CacheLoader.versionNode,
         PortalCMSCacheLoader.VERSION_NODE_NODE, hibernateJndiName);
      Thread versionNodeLoader = new Thread(versionNode);
      CacheLoader wspRef = new CacheLoader(CacheLoader.wspRef,
         PortalCMSCacheLoader.WSP_REF_NODE, hibernateJndiName);
      Thread wspRefLoader = new Thread(wspRef);
      CacheLoader versionRef = new CacheLoader(CacheLoader.versionRef,
         PortalCMSCacheLoader.VERSION_REF_NODE, hibernateJndiName);
      Thread versionRefLoader = new Thread(versionRef);

      wspPropLoader.start();
      versionPropLoader.start();
      wspNodeLoader.start();
      versionNodeLoader.start();
      wspRefLoader.start();
      versionRefLoader.start();

      // hang out here till the cache loader threads have pre-loaded the busy
      // nodes
      // of the CMS...don't allow the usage of CMS till this operation is
      // completed.
      while (!wspProp.done || !versionProp.done || !wspNode.done
         || !versionNode.done || !wspRef.done || !versionRef.done)
      {
         Thread.currentThread().sleep((long)2000);
      }

      JBossCachePersistenceManager.preloaded = true;
   }

   private static class CacheLoader implements Runnable
   {
      /**
       *
       */
      private static final String wspProp = "from org.jboss.portal.cms.hibernate.state.WSPProp";

      private static final String versionProp = "from org.jboss.portal.cms.hibernate.state.VersionProp";

      private static final String wspNode = "from org.jboss.portal.cms.hibernate.state.WSPNode";

      private static final String versionNode = "from org.jboss.portal.cms.hibernate.state.VersionNode";

      private static final String wspRef = "from org.jboss.portal.cms.hibernate.state.WSPRefs";

      private static final String versionRef = "from org.jboss.portal.cms.hibernate.state.VersionRefs";

      /**
       *
       */
      private String nodeQuery = null;

      private String nodePrefix = null;

      private boolean done = false;
      
      private String hibernateJndiName = null;

      private CacheLoader(String nodeQuery, String nodePrefix, String hibernateJndiName)
      {
         this.nodeQuery = nodeQuery;
         this.nodePrefix = nodePrefix;
         this.hibernateJndiName = hibernateJndiName;
      }

      public void run()
      {
         Session session = null;
         Transaction tx = null;
         try
         {
            session = HibernateUtil.getSessionFactory(this.hibernateJndiName).openSession();
            tx = session.beginTransaction();
            List rs = session.createQuery(this.nodeQuery).list();
            if (rs != null)
            {
               for (Iterator itr = rs.iterator(); itr.hasNext();)
               {
                  Base cour = (Base)itr.next();
                  cour.disableCacheItemPersistence();

                  if (cour instanceof WSPProp)
                  {
                     String node = this.nodePrefix
                        + "/"
                        + PortalCMSCacheLoader
                        .parseNodeName(((WSPProp)cour).getPropId());
                     pmCache.put(node, ((WSPProp)cour).getPropId(), cour);
                  }
                  else if (cour instanceof VersionProp)
                  {
                     String node = this.nodePrefix
                        + "/"
                        + PortalCMSCacheLoader
                        .parseNodeName(((VersionProp)cour)
                           .getPropId());
                     pmCache.put(node, ((VersionProp)cour).getPropId(), cour);
                  }
                  else if (cour instanceof WSPNode)
                  {
                     String node = this.nodePrefix
                        + "/"
                        + PortalCMSCacheLoader
                        .parseNodeName(((WSPNode)cour).getNodeId());
                     pmCache.put(node, ((WSPNode)cour).getNodeId(), cour);
                  }
                  else if (cour instanceof VersionNode)
                  {
                     String node = this.nodePrefix
                        + "/"
                        + PortalCMSCacheLoader
                        .parseNodeName(((VersionNode)cour)
                           .getNodeId());
                     pmCache.put(node, ((VersionNode)cour).getNodeId(), cour);
                  }
                  else if (cour instanceof WSPRefs)
                  {
                     String node = this.nodePrefix
                        + "/"
                        + PortalCMSCacheLoader
                        .parseNodeName(((WSPRefs)cour).getRefId());
                     pmCache.put(node, ((WSPRefs)cour).getRefId(), cour);
                  }
                  else if (cour instanceof VersionRefs)
                  {
                     String node = this.nodePrefix
                        + "/"
                        + PortalCMSCacheLoader
                        .parseNodeName(((VersionRefs)cour).getRefId());
                     pmCache.put(node, ((VersionRefs)cour).getRefId(), cour);
                  }
               }
            }
            log.info("Pre-loading the PersistenceManager Cache for"
               + this.nodePrefix + " (finished).......");
            tx.commit();
         }
         catch (CacheException ce)
         {
        	if(tx != null)
        	{
        		tx.rollback();
        	}
            throw new RuntimeException(ce);
         }
         finally
         {
            if(session != null && session.isOpen())
            {
            	session.close();
            }
            this.done = true;
         }
      }
   }

   // ----------actions for node
   // entities-------------------------------------------------------------------------------------------------------------
   /*
    * 
    */
   public boolean exists(NodeId node) throws ItemStateException
   {
      try
      {
         boolean exists = false;
         String nodeId = node.toString();
         if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.WSP_NODE_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            if (o != null)
            {
               exists = true;
            }
         }
         else if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.VERSION_NODE_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            if (o != null)
            {
               exists = true;
            }
         }
         return exists;
      }
      catch (CacheException ce)
      {
         throw new RuntimeException(ce);
      }
   }

   /*
    * 
    */
   public NodeState load(NodeId node) throws NoSuchItemStateException,
      ItemStateException
   {
      try
      {
         NodeState nodeState = null;
         String nodeId = node.toString();

         // get the nodeData
         byte[] nodeData = null;
         if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.WSP_NODE_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            nodeData = ((WSPNode)o).getData();
         }
         else if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.VERSION_NODE_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            nodeData = ((VersionNode)o).getData();
         }

         // parse propertyData into propertyState
         if (nodeData != null)
         {
            Session session = HibernateUtil.getSessionFactory(this.jndiName).getCurrentSession();
            session.beginTransaction();
            try
            {
               InputStream in = new ByteArrayInputStream(nodeData);

               // setup the propertyState
               nodeState = createNew(node);
               Serializer.deserialize(nodeState, in);
               IOTools.safeClose(in);
            }
            catch (Exception e)
            {
               session.getTransaction().rollback();
               
               if (e instanceof NoSuchItemStateException)
               {
                  throw (NoSuchItemStateException)e;
               }
               String msg = "failed to read node state from cache: "
                  + node.toString();
               log.error(msg, e);
               throw new ItemStateException(msg, e);
            }            
         }

         return nodeState;
      }
      catch (CacheException ce)
      {
         throw new RuntimeException(ce);
      }
   }

   /**
    *
    *
    */
   private void store(NodeState state) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      ByteArrayOutputStream out = new ByteArrayOutputStream(INITIAL_BUFFER_SIZE);
      try
      {
         Serializer.serialize(state, out);
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            VersionNode versionNode = new VersionNode(state.getId().toString(),
               out.toByteArray());

            // place this in the cache
            versionNode.resetCacheItemPersistence();
            String cacheNode = PortalCMSCacheLoader.VERSION_NODE_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(versionNode.getNodeId());
            pmCache.put(cacheNode, versionNode.getNodeId(), versionNode);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {

            WSPNode wspNode = new WSPNode(state.getId().toString(), out
               .toByteArray());

            // place this in the cache
            wspNode.resetCacheItemPersistence();
            String cacheNode = PortalCMSCacheLoader.WSP_NODE_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(wspNode.getNodeId());
            pmCache.put(cacheNode, wspNode.getNodeId(), wspNode);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to write node state: " + state.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
      finally
      {
         try
         {
            out.close();
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   /**
    *
    *
    */
   private void update(NodeState state) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      ByteArrayOutputStream out = new ByteArrayOutputStream(INITIAL_BUFFER_SIZE);
      try
      {
         Serializer.serialize(state, out);
         String id = state.getId().toString();
         String cacheNodeName = PortalCMSCacheLoader.parseNodeName(id);
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            VersionNode versionNode = (VersionNode)pmCache.get(
               PortalCMSCacheLoader.VERSION_NODE_NODE + "/" + cacheNodeName,
               id);
            if (versionNode == null)
            {
               throw new Exception("No such Node: " + state.getId());
            }
            versionNode.setData(out.toByteArray());
            versionNode.resetCacheItemPersistence();
            pmCache.put(PortalCMSCacheLoader.VERSION_NODE_NODE + "/"
               + cacheNodeName, id, versionNode);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            WSPNode wspNode = (WSPNode)pmCache.get(
               PortalCMSCacheLoader.WSP_NODE_NODE + "/" + cacheNodeName, id);
            if (wspNode == null)
            {
               throw new Exception("No such Node: " + state.getId());
            }
            wspNode.setData(out.toByteArray());
            wspNode.resetCacheItemPersistence();
            pmCache.put(PortalCMSCacheLoader.WSP_NODE_NODE + "/"
               + cacheNodeName, id, wspNode);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to write node state: " + state.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
      finally
      {
         try
         {
            out.close();
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   /** This is called by store(Changelog) this should not be called anywhere else */
   private void destroy(NodeState state) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      String id = state.getId().toString();
      String cacheNodeName = PortalCMSCacheLoader.parseNodeName(id);
      try
      {
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            pmCache.remove(PortalCMSCacheLoader.VERSION_NODE_NODE + "/"
               + cacheNodeName, id);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            pmCache.remove(PortalCMSCacheLoader.WSP_NODE_NODE + "/"
               + cacheNodeName, id);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to delete node state: " + state.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
   }

   // ----------actions for property
   // entities-------------------------------------------------------------------------------------------------------------
   /*
    * 
    */
   public boolean exists(PropertyId property) throws ItemStateException
   {
      try
      {
         boolean exists = false;
         String propId = property.toString();
         if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.WSP_PROP_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(propId), propId);
            if (o != null)
            {
               exists = true;
            }
         }
         else if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.VERSION_PROP_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(propId), propId);
            if (o != null)
            {
               exists = true;
            }
         }
         return exists;
      }
      catch (CacheException ce)
      {
         throw new RuntimeException(ce);
      }
   }

   /*
    * 
    */
   public PropertyState load(PropertyId property)
      throws NoSuchItemStateException, ItemStateException
   {
      try
      {
         PropertyState propertyState = null;
         String propId = property.toString();

         // get the propertyData
         byte[] propertyData = null;
         if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.WSP_PROP_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(propId), propId);
            propertyData = ((WSPProp)o).getData();
         }
         else if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.VERSION_PROP_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(propId), propId);
            propertyData = ((VersionProp)o).getData();
         }

         // parse propertyData into propertyState
         if (propertyData != null)
         {
            Session session = HibernateUtil.getSessionFactory(this.jndiName).getCurrentSession();
            session.beginTransaction();
            try
            {               
               InputStream in = new ByteArrayInputStream(propertyData);

               // setup the propertyState
               propertyState = createNew(property);
               Serializer.deserialize(propertyState, in, blobStore);
               IOTools.safeClose(in);
            }
            catch (Exception e)
            {
               session.getTransaction().rollback();
               
               if (e instanceof NoSuchItemStateException)
               {
                  throw (NoSuchItemStateException)e;
               }
               String msg = "failed to read property state from cache: "
                  + property.toString();
               log.error(msg, e);
               throw new ItemStateException(msg, e);
            }            
         }

         return propertyState;
      }
      catch (CacheException ce)
      {
         throw new RuntimeException(ce);
      }
   }

   /**
    *
    */
   private void store(PropertyState state) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      ByteArrayOutputStream out = new ByteArrayOutputStream(INITIAL_BUFFER_SIZE);
      try
      {
         Serializer.serialize(state, out, blobStore);
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            VersionProp versionProp = new VersionProp(state.getId().toString(),
               out.toByteArray());

            // place this in the cache
            versionProp.resetCacheItemPersistence();
            String cacheNode = PortalCMSCacheLoader.VERSION_PROP_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(versionProp.getPropId());
            pmCache.put(cacheNode, versionProp.getPropId(), versionProp);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {

            WSPProp wspProp = new WSPProp(state.getId().toString(), out
               .toByteArray());

            // place this in the cache
            wspProp.resetCacheItemPersistence();
            String cacheNode = PortalCMSCacheLoader.WSP_PROP_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(wspProp.getPropId());
            pmCache.put(cacheNode, wspProp.getPropId(), wspProp);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to write property state: " + state.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
      finally
      {
         try
         {
            out.close();
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   /**
    *
    *
    */
   private void update(PropertyState state) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      ByteArrayOutputStream out = new ByteArrayOutputStream(INITIAL_BUFFER_SIZE);
      try
      {
         Serializer.serialize(state, out, blobStore);
         String id = state.getId().toString();
         String cacheNodeName = PortalCMSCacheLoader.parseNodeName(id);
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            VersionProp versionProp = (VersionProp)pmCache.get(
               PortalCMSCacheLoader.VERSION_PROP_NODE + "/" + cacheNodeName,
               id);
            if (versionProp == null)
            {
               throw new Exception("No such Property: " + state.getId());
            }
            versionProp.setData(out.toByteArray());
            versionProp.resetCacheItemPersistence();
            pmCache.put(PortalCMSCacheLoader.VERSION_PROP_NODE + "/"
               + cacheNodeName, id, versionProp);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            WSPProp wspProp = (WSPProp)pmCache.get(
               PortalCMSCacheLoader.WSP_PROP_NODE + "/" + cacheNodeName, id);
            if (wspProp == null)
            {
               throw new Exception("No such Property: " + state.getId());
            }
            wspProp.setData(out.toByteArray());
            wspProp.resetCacheItemPersistence();
            pmCache.put(PortalCMSCacheLoader.WSP_PROP_NODE + "/"
               + cacheNodeName, id, wspProp);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to write property state: " + state.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
      finally
      {
         try
         {
            out.close();
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   /** This is called by store(Changelog) this should not be called anywhere else */
   private void destroy(PropertyState state) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      // make sure binary values (BLOBs) are properly removed
      InternalValue[] values = state.getValues();
      if (values != null)
      {
         for (int i = 0; i < values.length; i++)
         {
            InternalValue val = values[i];
            if (val != null)
            {
               if (val.getType() == PropertyType.BINARY)
               {
                  BLOBFileValue blobVal = (BLOBFileValue)val.internalValue();
                  // delete internal resource representation of BLOB value
                  blobVal.delete(true);
                  // also remove from BLOBStore
                  String blobId = blobStore.createId(
                     (PropertyId)state.getId(), i);
                  try
                  {
                     blobStore.remove(blobId);
                  }
                  catch (Exception e)
                  {
                     log.warn("failed to remove from BLOBStore: " + blobId, e);
                  }
               }
            }
         }
      }

      // remove the property node
      String id = state.getId().toString();
      String cacheNodeName = PortalCMSCacheLoader.parseNodeName(id);
      try
      {
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            pmCache.remove(PortalCMSCacheLoader.VERSION_PROP_NODE + "/"
               + cacheNodeName, id);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            pmCache.remove(PortalCMSCacheLoader.WSP_PROP_NODE + "/"
               + cacheNodeName, id);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to delete property state: " + state.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
   }

   // ----------actions for nodereferences
   // entities-------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public boolean exists(NodeReferencesId targetId) throws ItemStateException
   {
      try
      {
         boolean exists = false;
         String nodeId = targetId.toString();
         if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.WSP_REF_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            if (o != null)
            {
               exists = true;
            }
         }
         else if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.VERSION_REF_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            if (o != null)
            {
               exists = true;
            }
         }
         return exists;
      }
      catch (CacheException ce)
      {
         throw new RuntimeException(ce);
      }
   }

   /**
    *
    */
   public NodeReferences load(NodeReferencesId targetId)
      throws NoSuchItemStateException, ItemStateException
   {
      try
      {
         NodeReferences refs = null;
         String nodeId = targetId.toString();

         // get the nodeData
         byte[] nodeData = null;
         if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.WSP_REF_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            if (o == null || !(o instanceof Base))
            {
               throw new NoSuchItemStateException(targetId.toString());
            }
            nodeData = ((WSPRefs)o).getData();
         }
         else if (this.schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            Object o = this.pmCache.get(PortalCMSCacheLoader.VERSION_REF_NODE
               + "/" + PortalCMSCacheLoader.parseNodeName(nodeId), nodeId);
            if (o == null || !(o instanceof Base))
            {
               throw new NoSuchItemStateException(targetId.toString());
            }
            nodeData = ((VersionRefs)o).getData();
         }

         // parse propertyData into propertyState
         if (nodeData != null)
         {
            Session session = HibernateUtil.getSessionFactory(this.jndiName).getCurrentSession();
            session.beginTransaction();
            try
            {
               InputStream in = new ByteArrayInputStream(nodeData);

               // setup the propertyState
               refs = new NodeReferences(targetId);
               Serializer.deserialize(refs, in);
               IOTools.safeClose(in);
            }
            catch (Exception e)
            {
               session.getTransaction().rollback();
               
               if (e instanceof NoSuchItemStateException)
               {
                  throw (NoSuchItemStateException)e;
               }
               String msg = "failed to read reference from cache: "
                  + targetId.toString();
               log.error(msg, e);
               throw new ItemStateException(msg, e);
            }            
         }

         return refs;
      }
      catch (CacheException ce)
      {
         throw new RuntimeException(ce);
      }
   }

   /**
    *
    *
    */
   private void store(NodeReferences refs) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      ByteArrayOutputStream out = new ByteArrayOutputStream(INITIAL_BUFFER_SIZE);
      try
      {
         Serializer.serialize(refs, out);
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            VersionRefs versionRefs = new VersionRefs(refs.getId().toString(),
               out.toByteArray());

            // place this in the cache
            versionRefs.resetCacheItemPersistence();
            String cacheNode = PortalCMSCacheLoader.VERSION_REF_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(versionRefs.getRefId());
            pmCache.put(cacheNode, versionRefs.getRefId(), versionRefs);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {

            WSPRefs wspRefs = new WSPRefs(refs.getId().toString(), out
               .toByteArray());

            // place this in the cache
            wspRefs.resetCacheItemPersistence();
            String cacheNode = PortalCMSCacheLoader.WSP_REF_NODE + "/"
               + PortalCMSCacheLoader.parseNodeName(wspRefs.getRefId());
            pmCache.put(cacheNode, wspRefs.getRefId(), wspRefs);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to write reference state: " + refs.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
      finally
      {
         try
         {
            out.close();
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   /**
    *
    */
   private void destroy(NodeReferences refs) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      String id = refs.getId().toString();
      String cacheNodeName = PortalCMSCacheLoader.parseNodeName(id);
      try
      {
         if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
         {
            pmCache.remove(PortalCMSCacheLoader.VERSION_REF_NODE + "/"
               + cacheNodeName, id);
         }
         else if (schemaObjectPrefix
            .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
         {
            pmCache.remove(PortalCMSCacheLoader.WSP_REF_NODE + "/"
               + cacheNodeName, id);
         }
      }
      catch (Exception e)
      {
         String msg = "failed to delete reference state: " + refs.getId();
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
   }

   // ----------------------------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public void close() throws Exception
   {
      // cleanup the jboss cache service
      if (pmCache != null)
      {
         pmCache.stopService();
         pmCache.destroyService();
         pmCache = null;
         preloaded = false;
      }

      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      try
      {
         if (externalBLOBs)
         {
            // close BLOB file system
            blobFS.close();
            blobFS = null;
         }
         blobStore = null;
      }
      finally
      {
         initialized = false;
      }
   }

   /** {@inheritDoc} */
   public NodeState createNew(NodeId id)
   {
      return new NodeState(id, null, null, NodeState.STATUS_NEW, false);
   }

   /** {@inheritDoc} */
   public PropertyState createNew(PropertyId id)
   {
      return new PropertyState(id, PropertyState.STATUS_NEW, false);
   }

   /**
    *
    */
   public void store(ChangeLog changeLog) throws ItemStateException
   {
      Session session = HibernateUtil.getSessionFactory(this.jndiName).getCurrentSession();
      session.beginTransaction();
      try
      {         
         storeHB(changeLog);
      }
      catch (ItemStateException e)
      {
    	 session.getTransaction().rollback();
         throw e;
      }      
   }

   public void storeHB(ChangeLog changeLog) throws ItemStateException
   {
      Iterator iter = changeLog.deletedStates();
      while (iter.hasNext())
      {
         ItemState state = (ItemState)iter.next();
         if (state.isNode())
         {
            destroy((NodeState)state);
         }
         else
         {
            destroy((PropertyState)state);
         }
      }
      iter = changeLog.addedStates();
      while (iter.hasNext())
      {
         ItemState state = (ItemState)iter.next();
         if (state.isNode())
         {
            store((NodeState)state);
         }
         else
         {
            store((PropertyState)state);
         }
      }
      iter = changeLog.modifiedStates();
      while (iter.hasNext())
      {
         ItemState state = (ItemState)iter.next();
         if (state.isNode())
         {
            update((NodeState)state);
         }
         else
         {
            update((PropertyState)state);
         }
      }
      iter = changeLog.modifiedRefs();
      while (iter.hasNext())
      {
         NodeReferences refs = (NodeReferences)iter.next();
         if (refs.hasReferences())
         {
            store(refs);
         }
         else
         {
            if (exists(refs.getId()))
            {
               destroy(refs);
            }
         }
      }
   }

   /**
    *
    */
   protected boolean exists(String blobid) throws ItemStateException
   {
      return exists(blobSelectExist, blobid);
   }

   /**
    *
    *
    */
   private boolean exists(String query, String id) throws ItemStateException
   {
      if (!initialized)
      {
         throw new IllegalStateException("not initialized");
      }

      Session session = HibernateUtil.getSessionFactory(this.jndiName).getCurrentSession();
      session.beginTransaction();
      try
      {
         List rs = session.createQuery(query).setString(0, id).list();
         Iterator iter = rs.iterator();
         return iter.hasNext();
      }
      catch (Exception e)
      {
    	 session.getTransaction().rollback();
    	 
         String msg = "failed to check existence of node state: " + id;
         log.error(msg, e);
         throw new ItemStateException(msg, e);
      }
   }

   // -------------------------------------------------< misc. helper methods >
   protected void logException(String message, SQLException se)
   {
      if (message != null)
      {
         log.error(message);
      }
      log.error("    reason: " + se.getMessage());
      log.error("state/code: " + se.getSQLState() + "/" + se.getErrorCode());
      log.debug("      dump:", se);
   }

   // --------------------------------------------------------< inner classes >
   class DbBLOBStore implements BLOBStore
   {

      /**
       *
       */
      public String createId(PropertyId id, int index)
      {
         // the blobId is a simple string concatenation of id plus index
         StringBuffer sb = new StringBuffer();
         sb.append(id.toString());
         sb.append('[');
         sb.append(index);
         sb.append(']');
         return sb.toString();
      }

      /**
       *
       */
      public InputStream get(String blobId) throws Exception
      {
         Session session = HibernateUtil.getSessionFactory(jndiName).getCurrentSession();
         session.beginTransaction();
         try
         {
            List rs = session.createQuery(blobSelectData).setString(0, blobId)
               .list();
            Iterator iter = rs.iterator();
            
            /*InputStream is = null;
            byte[] blob = (byte[])iter.next();
            is = new ByteArrayInputStream(blob);*/
            
            /**
             * This needs to be done this way due to issues with a combination of Postgresql driver handling blobs and Jackrabbit 1.4
             * 
             * Its fine for: hsqldb, mysql5, and oracle10g
             */
            InputStream is = null;            
            java.sql.Blob blob = (java.sql.Blob)iter.next();
            InputStream blobStream = blob.getBinaryStream();
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024]; //using a 1K buffer
            int bytesRead = -1;
            while((bytesRead=blobStream.read(buffer)) != -1)
            {
               bos.write(buffer, 0, bytesRead);
               bos.flush();
            }            
            try{blobStream.close();}catch(Exception e){}            
            is = new ByteArrayInputStream(bos.toByteArray());
            
            return is;
         }
         catch (Exception e)
         {
        	session.getTransaction().rollback();
        	
            if (e instanceof NoSuchItemStateException)
            {
               throw (NoSuchItemStateException)e;
            }
            String msg = "failed to read binary data: " + blobId;
            log.error(msg, e);
            throw new ItemStateException(msg, e);
         }
      }

      /**
       *
       */
      public void put(String blobId, InputStream in, long size)
         throws Exception
      {
         boolean update = exists(blobId);

         Session session = HibernateUtil.getSessionFactory(jndiName).getCurrentSession();
         session.beginTransaction();
         if (update)
         {
            try
            {

               Query query = session.createQuery(blobSelect);
               if (schemaObjectPrefix
                  .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
               {
                  query.setString(0, blobId);
                  VersionBinVal versionNode = (VersionBinVal)query
                     .uniqueResult();
                  if (versionNode == null)
                  {
                     throw new Exception("No such Node: " + blobId);
                  }
                  versionNode.setId(blobId);
                  
                  //versionNode.setData(IOTools.getBytes(in));
                  versionNode.setData(Hibernate.createBlob(in));
                  session.flush();
               }
               else if (schemaObjectPrefix
                  .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
               {
                  query.setString(0, blobId);
                  WSPBinVal wspNode = (WSPBinVal)query.uniqueResult();
                  if (wspNode == null)
                  {
                     throw new Exception("No such Node: " + blobId);
                  }                  
                  wspNode.setId(blobId);
                  
                  //wspNode.setData(IOTools.getBytes(in));
                  wspNode.setData(Hibernate.createBlob(in));
                  session.flush();
               }
            }
            catch (Exception e)
            {
               session.getTransaction().rollback();
               
               String msg = "failed to write node state: " + blobId;
               log.error(msg, e);
               throw new ItemStateException(msg, e);
            }            
         }
         else
         // insert
         {
            try
            {

               if (schemaObjectPrefix
                  .equalsIgnoreCase(HibernateStoreConstants.versionPrefix))
               {
                  //VersionBinVal versionNode = new VersionBinVal(blobId,
                  //		  IOTools.getBytes(in));
                  VersionBinVal versionNode = new VersionBinVal(blobId,
                		  Hibernate.createBlob(in));
                  
                  session.save(versionNode);
                  session.flush();
               }
               else if (schemaObjectPrefix
                  .equalsIgnoreCase(HibernateStoreConstants.wspPrefix))
               {
                  //WSPBinVal wspNode = new WSPBinVal(blobId, IOTools.getBytes(in));
                  WSPBinVal wspNode = new WSPBinVal(blobId, Hibernate.createBlob(in));
                  
                  session.save(wspNode);
                  session.flush();
               }
            }
            catch (Exception e)
            {
               session.getTransaction().rollback();
               
               String msg = "failed to write node state: " + blobId;
               log.error(msg, e);
               throw new ItemStateException(msg, e);
            }            
         }
      }

      /**
       *
       */
      // public synchronized boolean remove(String blobId) throws Exception
      public boolean remove(String blobId) throws Exception
      {
         Session session = HibernateUtil.getSessionFactory(jndiName).getCurrentSession();
         session.beginTransaction();
         try
         {
            Query query = session.createQuery(nodeBinValSelect).setString(0,
               blobId);
            Object result = query.uniqueResult();
            if (result != null)
            {
               session.delete(result);
            }
            return true;
         }
         catch (Exception e)
         {
        	session.getTransaction().rollback();
        	
            String msg = "failed to delete binval: " + blobId;
            log.error(msg, e);
            throw new ItemStateException(msg, e);
         }         
      }
   }
}
