/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.apache.log4j.Logger;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.jboss.portal.cms.CMSException;
import org.jboss.portal.cms.impl.ContentImpl;
import org.jboss.portal.cms.impl.FileImpl;
import org.jboss.portal.cms.impl.FolderImpl;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.model.Content;
import org.jboss.portal.cms.model.Folder;
import org.jboss.portal.cms.util.FileUtil;
import org.jboss.portal.cms.util.NodeUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.Enumeration;
import java.util.Locale;
import java.util.StringTokenizer;

/**
 * Saves an uploaded archive to the repo.
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:sshah@redhat.com">Sohil Shah</a>
 */
public class StoreArchiveCommand extends JCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 1568453649437987499L;
   private static Logger log = Logger.getLogger(StoreArchiveCommand.class);

   String msRootPath;
   byte[] archiveBytes;
   String msLanguage;

   /**
    * @param sRootPath
    * @param sLanguage
    */
   public StoreArchiveCommand(String sRootPath, byte[] archiveBytes, String sLanguage)
   {
      this.msRootPath = sRootPath;
      this.archiveBytes = archiveBytes;
      this.msLanguage = sLanguage;
   }

   /**
    *
    */
   public Object execute()
   {
      File tmpFile = null;
      try
      {
         tmpFile = this.getZipFile();

         ZipFile zipFile = new ZipFile(tmpFile);
         ZipEntry zipEntry;
         Enumeration entries = zipFile.getEntries();
         while (entries.hasMoreElements())
         {
            zipEntry = (ZipEntry)entries.nextElement();

            String itemName = zipEntry.getName();
            if (!NodeUtil.CHECK_FOR_XSS_PATTERN.matcher(itemName).matches())
            {
               log.info("Zip file: '" + itemName + "' is not a valid file name. It will be skipped.");
            }
            else
            {
               if (!zipEntry.isDirectory())
               {
                  this.addFile(zipFile, zipEntry);
               }
               else // isDirectory
               {
                  this.addFolder(zipEntry);
               }
            }
         }

         return null;
      }
      catch (Exception e)
      {
         throw new CMSException(e);
      }
      finally
      {
         if (tmpFile != null)
         {
            tmpFile.delete();
         }
      }
   }

   /**
    * @param zipFile
    * @param zipEntry
    * @throws IOException
    */
   private void addFile(ZipFile zipFile, ZipEntry zipEntry) throws IOException
   {
      try
      {
         String itemName = zipEntry.getName();
         long fileSize = zipEntry.getSize();
         byte[] zipBytes = new byte[(int)fileSize];
         InputStream zipDataStream = zipFile.getInputStream(zipEntry);
         long bytesRead = 0;

         while (bytesRead < fileSize)
         {
            bytesRead += zipDataStream.read(zipBytes, (int)bytesRead, (int)(fileSize - bytesRead));
         }

         org.jboss.portal.cms.model.File file = new FileImpl();

         String sBasePath = FileUtil.cleanDoubleSlashes("/" + this.msRootPath + "/" + itemName);
         sBasePath = FileUtil.cleanDoubleSlashes(sBasePath); // hackish, but necessary for root path uploads.
         file.setBasePath(sBasePath);

         Content content = new ContentImpl();
         content.setEncoding("UTF-8");
         content.setTitle(itemName);
         content.setDescription(itemName);
         content.setBasePath(sBasePath + "/" + this.msLanguage);
         content.setBytes(zipBytes);
         file.setContent(new Locale(this.msLanguage), content);

         //Store the file
         JCRCommand nodeExists = (JCRCommand)context.getCommandFactory().createItemExistsCommand(file.getBasePath());
         Boolean bExists = (Boolean)context.execute(nodeExists);
         if (!bExists.booleanValue())
         {
            //Create new file
            JCRCommand newFile = (JCRCommand)context.getCommandFactory().createNewFileCommand(file, content);
            context.execute(newFile);
         }
         else
         {
            //Update the existing one, and make the new version created 
            //in return as live
            JCRCommand updateFile = (JCRCommand)context.getCommandFactory().createUpdateFileCommand(file, content, true);
            context.execute(updateFile);
         }
      }
      catch (CMSException e)
      {
         //Log the error, but don't fail creation of other resources in the archive
         log.error(this, e);
      }
   }

   private void addFolder(ZipEntry zipEntry)
   {
      try
      {
         String itemName = zipEntry.getName();

         //trim trailing slash.
         if (itemName.endsWith("/"))
         {
            itemName = itemName.substring(0, itemName.length() - 1);
         }

         if (!"".equals(itemName))
         {
            String sBasePath = FileUtil.cleanDoubleSlashes("/" + this.msRootPath + "/" + itemName);
            sBasePath = FileUtil.cleanDoubleSlashes(sBasePath); // hackish, but necessary for root path uploads.
            String sParentPath = NodeUtil.getParentPath(sBasePath);
            JCRCommand nodeExists = (JCRCommand)context.getCommandFactory().createItemExistsCommand(sParentPath);
            Boolean bExists = (Boolean)context.execute(nodeExists);
            if (!bExists.booleanValue())
            {
               this.createParentHierarchy(sParentPath);
            }

            JCRCommand nodeExists2 = (JCRCommand)context.getCommandFactory().createItemExistsCommand(sBasePath);
            Boolean bExists2 = (Boolean)context.execute(nodeExists2);
            if (!bExists2.booleanValue())
            {
               Folder folder = new FolderImpl();
               folder.setName(itemName);
               folder.setDescription(itemName);
               folder.setTitle(itemName);
               folder.setLastModified(new Date());
               folder.setBasePath(sBasePath);

               JCRCommand folderSave = (JCRCommand)context.getCommandFactory().createFolderSaveCommand(folder);
               context.execute(folderSave);
            }
         }
      }
      catch (Exception e)
      {
         //Log the error, but don't fail creation of other resources in the archive
         log.error(this, e);
      }
   }

   /** @param parentPath  */
   private void createParentHierarchy(String parentPath) throws CMSException
   {
      StringTokenizer tokenizer = new StringTokenizer(parentPath, "/");

      StringBuffer buffer = new StringBuffer("/");
      while (tokenizer.hasMoreTokens())
      {
         buffer.append(tokenizer.nextToken());
         String cour = buffer.toString();

         JCRCommand nodeExists = (JCRCommand)context.getCommandFactory().createItemExistsCommand(cour);
         Boolean bExists = (Boolean)context.execute(nodeExists);
         if (!bExists.booleanValue())
         {
            this.createFolder(cour);
         }

         if (tokenizer.hasMoreTokens())
         {
            buffer.append("/");
         }
      }
   }

   /**
    * @param folderPath
    * @throws CMSException
    */
   private void createFolder(String folderPath) throws CMSException
   {
      Folder folder = new FolderImpl();
      folder.setName(folderPath);
      folder.setDescription(folderPath);
      folder.setTitle(folderPath);
      folder.setLastModified(new Date());
      folder.setBasePath(folderPath);

      JCRCommand parentSave = (JCRCommand)context.getCommandFactory().createFolderSaveCommand(folder);
      context.execute(parentSave);
   }

   /** @return  */
   private File getZipFile() throws Exception
   {
      File zipFile = null;

      zipFile = File.createTempFile("jbportal_", "_cmsimport.zip");
      FileOutputStream fos = new FileOutputStream(zipFile.getCanonicalPath());
      try
      {
         fos.write(this.archiveBytes);
         fos.close();
      }
      finally
      {
         fos.close();
      }

      return zipFile;
   }
}