/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.workflow;

import org.apache.log4j.Logger;
import org.jboss.portal.common.io.IOTools;
import org.jboss.portal.identity.IdentityContext;
import org.jboss.portal.identity.IdentityServiceController;
import org.jboss.portal.identity.MembershipModule;
import org.jboss.portal.identity.Role;
import org.jboss.portal.identity.UserModule;
import org.jboss.portal.jems.as.JNDI;
import org.jboss.portal.jems.as.system.AbstractJBossService;
import org.jboss.portal.workflow.WorkflowException;
import org.jboss.portal.workflow.service.WorkflowService;
import org.jbpm.JbpmContext;
import org.jbpm.db.GraphSession;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;
import org.jbpm.taskmgmt.exe.TaskInstance;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;


/**
 * Created on : Dec 19, 2006
 *
 * @author Sohil Shah - sohil.shah@jboss.com
 */
public class ApprovePublishImpl extends AbstractJBossService implements ApprovePublish
{
   /**
    *
    */
   private static final Logger log = Logger.getLogger(ApprovePublishImpl.class);

   /**
    *
    */
   private WorkflowService workflowService = null;
   private IdentityServiceController identityServiceController = null;
   private String process = null;
   private String processName = null;
   private String managerRoles = null;
   private String[] managers = null;
   private Set<String> managerSet = null;
   private boolean overwrite = false;
   private String from = null;
   private String subject = null;
   private String body = null;
   private JNDI.Binding jndiBinding;

   private MembershipModule membershipModule = null;
   private UserModule userModule = null;

   private String jndiName = null;

   /**
    *
    *
    */
   public ApprovePublishImpl()
   {

   }

   /**
    *
    */
   public void startService() throws Exception
   {
      super.startService();

      if (this.jndiName != null)
      {
         jndiBinding = new JNDI.Binding(jndiName, this);
         jndiBinding.bind();
      }

      InputStream is = null;
      JbpmContext jbpmContext = null;
      try
      {
         is = new ByteArrayInputStream(this.process.getBytes());
         jbpmContext = this.workflowService.getJbpmConfiguration().createJbpmContext();

         Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(is);

         Element root = document.getDocumentElement();
         this.processName = root.getAttribute("name");

         ProcessDefinition processDefinition = jbpmContext.getGraphSession().findLatestProcessDefinition(this.processName);
         if (processDefinition == null)
         {
            processDefinition = ProcessDefinition.parseXmlString(this.process);
            jbpmContext.deployProcessDefinition(processDefinition);
         }
         else
         {
            // A process definition already exists....should deploy a new version
            // of the definition if overwrite is true
            ProcessDefinition fromConfig = ProcessDefinition.parseXmlString(this.process);
            if (this.overwrite)
            {
               // If the two of them are not same, create a new version
               // of this process definition
               jbpmContext.deployProcessDefinition(fromConfig);
            }
         }
      }
      catch (Exception e)
      {
         //
         this.stopService();

         //
         throw e;
      }
      finally
      {
         IOTools.safeClose(is);
         IOTools.safeClose(jbpmContext);
      }

      //process managers that can serve as approvers/rejecters
      StringTokenizer st = new StringTokenizer(this.managerRoles, ",");
      this.managers = new String[st.countTokens()];
      this.managerSet = new HashSet<String>();
      for (int i = 0; i < managers.length; i++)
      {
         this.managers[i] = st.nextToken().trim();
         this.managerSet.add(this.managers[i]);
      }

      this.membershipModule = (MembershipModule)identityServiceController.getIdentityContext().
              getObject(IdentityContext.TYPE_MEMBERSHIP_MODULE);

      this.userModule = (UserModule)identityServiceController.getIdentityContext().
              getObject(IdentityContext.TYPE_USER_MODULE);

   }

   /**
    *
    */
   public void stopService() throws Exception
   {
      super.stopService();

      if (jndiBinding != null)
      {
         jndiBinding.unbind();
         jndiBinding = null;
      }
   }

   /** @return  */
   public WorkflowService getWorkflowService()
   {
      return this.workflowService;
   }

   /** @param workflowService  */
   public void setWorkflowService(WorkflowService workflowService)
   {
      this.workflowService = workflowService;
   }

   /** @return  */
   public String getProcess()
   {
      return this.process;
   }

   /** @param process  */
   public void setProcess(String process)
   {
      this.process = process;
   }


   /** @return the managerRoles */
   public String getManagerRoles()
   {
      return managerRoles;
   }

   /** @param managerRoles the manager managerRoles to set */
   public void setManagerRoles(String managerRoles)
   {
      this.managerRoles = managerRoles;
   }


   /** @return the overwrite */
   public boolean isOverwrite()
   {
      return overwrite;
   }

   /** @param overwrite the overwrite to set */
   public void setOverwrite(boolean overwrite)
   {
      this.overwrite = overwrite;
   }

   /** @return  */
   public String getJNDIName()
   {
      return this.jndiName;
   }

   /** @param jndiName  */
   public void setJNDIName(String jndiName)
   {
      this.jndiName = jndiName;
   }

   /**
    *
    */
   public IdentityServiceController getIdentityServiceController()
   {
      return identityServiceController;
   }

   /** @param identityServiceController  */
   public void setIdentityServiceController(
           IdentityServiceController identityServiceController)
   {
      this.identityServiceController = identityServiceController;
   }


   /** @return the body */
   public String getBody()
   {
      return body;
   }

   /** @param body the body to set */
   public void setBody(String body)
   {
      this.body = body;
   }

   /** @return the from */
   public String getFrom()
   {
      return from;
   }

   /** @param from the from to set */
   public void setFrom(String from)
   {
      this.from = from;
   }

   /** @return the subject */
   public String getSubject()
   {
      return subject;
   }

   /** @param subject the subject to set */
   public void setSubject(String subject)
   {
      this.subject = subject;
   }

   //----------ApprovePublish Implementation------------------------------------------------------------------
   /**
    * Called when content is added to the CMS, and needs to be approved by the managers before it can be published to go
    * live
    *
    * @param content
    * @return returns the process id of the workflow process set in motion
    */
   public long requestApproval(Content content) throws WorkflowException
   {
      long processId = 0;
      JbpmContext jbpmContext = null;
      ProcessInstance processInstance = null;
      boolean success = false;
      try
      {
         jbpmContext = this.workflowService.getJbpmConfiguration().createJbpmContext();

         //The next line creates one execution of the process definition.
         // After construction, the process execution has one main path
         // of execution (=the root token) that is positioned in the
         // start-state.
         processInstance = jbpmContext.newProcessInstance(this.processName);

         //After construction, the process execution has one main path
         // of execution (=the root token).
         Token token = processInstance.getRootToken();

         //set the process variables
         processInstance.getContextInstance().setVariable("content", content);
         processInstance.getContextInstance().setVariable("managers", this.managers);
         processInstance.getContextInstance().setVariable("from", this.from);
         processInstance.getContextInstance().setVariable("subject", this.subject);
         processInstance.getContextInstance().setVariable("body", this.body);

         //start the workflow, starts the cms publish approval workflow
         //this creates a task to approve/reject a cms publish for the Admins
         token.signal();

         //mark as a successfull process initiation
         success = true;
      }
      catch (Exception e)
      {
         success = false;
         throw new WorkflowException(e);
      }
      finally
      {
         if (processInstance != null && success)
         {
            jbpmContext.save(processInstance);
            processId = processInstance.getId();
         }
         IOTools.safeClose(jbpmContext);
      }
      return processId;
   }

   public void processManagerResponse(long processId, String manager, boolean approved) throws WorkflowException
   {
      JbpmContext jbpmContext = null;
      ProcessInstance processInstance = null;
      boolean isManager = false;
      try
      {
         jbpmContext = this.workflowService.getJbpmConfiguration().createJbpmContext();

         //Now, we search for all process instances of this process definition.
         processInstance = jbpmContext.loadProcessInstance(processId);

         if (processInstance.hasEnded())
         {
            log.debug("This process has already ended...");
            return;
         }

         processInstance.getContextInstance().setVariable("approved", new Boolean(approved));

         Collection allTasks = processInstance.getTaskMgmtInstance().getTaskInstances();
         if (allTasks != null)
         {
            for (Iterator itr = allTasks.iterator(); itr.hasNext();)
            {
               TaskInstance cour = (TaskInstance)itr.next();
               if (this.isManager(manager, cour.getActorId()))
               {
                  isManager = true;
                  log.debug("Manager=" + cour.getActorId() + "(" + processId + ")");

                  //check and make sure this task instance is not marked for deletion
                  if (cour.getVariable(processInstance.getId() + ":" + cour.getId()) != null)
                  {
                     continue;
                  }

                  if (!approved)
                  {
                     cour.start();
                     cour.end("rejection");
                     break;
                  }
                  else
                  {
                     cour.start();
                     cour.end("approval");
                     break;
                  }
               }
            }
         }
      }
      catch (Exception e)
      {
         throw new WorkflowException(e);
      }
      finally
      {
         if (processInstance != null)
         {
            jbpmContext.save(processInstance);
         }

         if (!isManager)
         {
            WorkflowException we = new WorkflowException("You are not authorized to Approve/Deny content publish requests");
            throw we;
         }
         IOTools.safeClose(jbpmContext);
      }
   }

   public void processManagerResponse(long processId, String manager, String modifiedContent) throws WorkflowException
   {
      JbpmContext jbpmContext = null;
      ProcessInstance processInstance = null;
      boolean isManager = false;
      try
      {
         jbpmContext = this.workflowService.getJbpmConfiguration().createJbpmContext();

         //Now, we search for all process instances of this process definition.
         processInstance = jbpmContext.loadProcessInstance(processId);

         if (processInstance.hasEnded())
         {
            log.debug("This process has already ended...");
            return;
         }

         processInstance.getContextInstance().setVariable("approved", new Boolean(true));
         processInstance.getContextInstance().setVariable("modifiedContent", modifiedContent.getBytes());

         @SuppressWarnings("unchecked")
         Collection<TaskInstance> allTasks = processInstance.getTaskMgmtInstance().getTaskInstances();
         if (allTasks != null)
         {
            for (TaskInstance currentTask : allTasks)
            {
               if (this.isManager(manager, currentTask.getActorId()))
               {
                  isManager = true;
                  log.debug("Manager=" + currentTask.getActorId() + "(" + processId + ")");

                  //check and make sure this task instance is not marked for deletion
                  if (currentTask.getVariable(processInstance.getId() + ":" + currentTask.getId()) != null)
                  {
                     continue;
                  }

                  currentTask.start();
                  currentTask.end("approval");
                  break;
               }
            }
         }
      }
      catch (Exception e)
      {
         throw new WorkflowException(e);
      }
      finally
      {
         if (processInstance != null)
         {
            jbpmContext.save(processInstance);
         }

         if (!isManager)
         {
            throw new WorkflowException("You are not authorized to Approve/Deny content publish requests");
         }
         IOTools.safeClose(jbpmContext);
      }
   }

   /**
    * Retrieves a queue of unapproved content associated with the specified file in the CMS
    *
    * @param filePath
    * @return
    * @throws WorkflowException
    */
   public Collection<Content> getPendingQueue(String filePath) throws WorkflowException
   {
      Collection<Content> pendingQueue = new ArrayList<Content>();
      JbpmContext jbpmContext = null;
      try
      {
         jbpmContext = this.workflowService.getJbpmConfiguration().createJbpmContext();

         GraphSession graphSession = jbpmContext.getGraphSession();
         ProcessDefinition processDef = graphSession.findLatestProcessDefinition(this.processName);

         @SuppressWarnings("unchecked")
         List<ProcessInstance> processInstances = graphSession.findProcessInstances(processDef.getId());

         if (processInstances != null)
         {
            for (ProcessInstance processInstance : processInstances)
            {
               //iterate through a list of currently pending approval tasks
               if (!processInstance.hasEnded())
               {
                  Content content = this.getContent(processInstance);

                  //apply proper criteria to extract pending content only for the specified file
                  if (content != null)
                  {
                     int lastIndex = content.getPath().lastIndexOf('/');
                     String criteriaPath = content.getPath().substring(0, lastIndex);

                     if (criteriaPath.trim().equals(filePath.trim()))
                     {
                        content.setProcessId(String.valueOf(processInstance.getId()));
                        pendingQueue.add(content);
                     }
                  }
               }
            }
         }
      }
      catch(Exception e)
      {
         log.error(this, e);
         throw new WorkflowException(e);
      }
      finally
      {
         IOTools.safeClose(jbpmContext);
      }
      return pendingQueue;
   }

   /**
    * Retrieves a queue of unapproved content for everything in the CMS.
    *
    * @return
    * @throws WorkflowException
    */
   public Collection<Content> getAllPendingInQueue() throws WorkflowException
   {
      Collection<Content> pendingQueue = new ArrayList<Content>();
      JbpmContext jbpmContext = null;
      try
      {
         jbpmContext = this.workflowService.getJbpmConfiguration().createJbpmContext();

         GraphSession graphSession = jbpmContext.getGraphSession();
         ProcessDefinition processDef = graphSession.findLatestProcessDefinition(this.processName);

         @SuppressWarnings("unchecked")
         List<ProcessInstance> processInstances = graphSession.findProcessInstances(processDef.getId());

         if (processInstances != null)
         {
            for (ProcessInstance processInstance : processInstances)
            {
               //iterate through a list of currently pending approval tasks
               if (!processInstance.hasEnded())
               {
                  Content content = this.getContent(processInstance);

                  //apply proper criteria to extract pending content only for the specified file
                  if (content != null)
                  {
                     content.setProcessId(String.valueOf(processInstance.getId()));
                     pendingQueue.add(content);
                  }
               }
            }
         }
      }
      catch(Exception e)
      {
         log.error(this, e);
         throw new WorkflowException(e);
      }
      finally
      {
         IOTools.safeClose(jbpmContext);
      }
      return pendingQueue;
   }

   /** @return  */
   public Set<String> getManagers()
   {
      return this.managerSet;
   }

   //----------------------------------------------------------------------------------------------------------------
   /**
    * checks to see if the user trying to approve/deny a publish request belongs to the approved list of managers
    *
    * @param user
    * @param managerRole
    * @return
    */
   private boolean isManager(String user, String managerRole) throws Exception
   {
      boolean isManager = false;

      Set userRoles = this.membershipModule.getRoles(
              this.userModule.findUserByUserName(user));
      if (userRoles != null)
      {
         for (Iterator itr = userRoles.iterator(); itr.hasNext();)
         {
            Role cour = (Role)itr.next();
            if (cour.getName().equalsIgnoreCase(managerRole))
            {
               //user is allowed to be a manager for this workflow
               isManager = true;
               break;
            }
         }
      }

      return isManager;
   }
   
   private Content getContent(ProcessInstance processInstance) throws Exception
   {
      Content content = null;
      
      Object object = processInstance.getContextInstance().getVariable("content");
      content = CMSWorkflowUtil.deserializeContent(object);
      
      return content;
   }
}
