/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.workflow;

import org.apache.log4j.Logger;
import org.jboss.portal.cms.CMS;
import org.jboss.portal.cms.Command;
import org.jboss.portal.cms.impl.interceptors.ACLInterceptor;
import org.jboss.portal.cms.impl.jcr.JCRCMS;
import org.jboss.portal.cms.model.File;
import org.jbpm.graph.def.ActionHandler;
import org.jbpm.graph.exe.ExecutionContext;

import java.util.List;
import java.util.Locale;
import java.util.ArrayList;

/**
 * Created on : Dec 20, 2006
 *
 * @author Sohil Shah - sohil.shah@jboss.com
 */
public class FinalizePublish implements ActionHandler
{
   /**
    *
    */
   private static Logger log = Logger.getLogger(FinalizePublish.class);

   /**
    *
    */
   public void execute(ExecutionContext executionContext)
   {
      Content content = this.readContent(executionContext);
      boolean approved = ((Boolean)executionContext.getContextInstance().getVariable("approved")).booleanValue();
      byte[] modifiedContent = (byte[])executionContext.getContextInstance().getVariable("modifiedContent");
      long processId = executionContext.getProcessInstance().getId();
      CMS cms = this.getCMSService();

      int lastIndex = content.getPath().lastIndexOf('/');
      String path = content.getPath().substring(0, lastIndex);
      String language = content.getPath().substring(lastIndex + 1);      
      if (approved)
      {
         ACLInterceptor.turnOff();
         JCRCMS.turnOffWorkflow();

         //create this content in the CMS and make it live
         Command command = cms.getCommandFactory().createContentGetVersionsCommand(content.getPath());
         List versions = (List)cms.execute(command);
         org.jboss.portal.cms.model.Content processContent = this.findProcessContent(processId, versions);

         /**
          * if processContent is null, a pending version of the content was not found
          * this could be due to an outdated link. let the process finish, but nothing
          * new will be published
          */
         if (processContent != null)
         {
            //get the file to be updated and published live
            command = cms.getCommandFactory().createFileGetCommand(path, processContent.getVersionNumber(),
               new Locale(language));
            File file = (File)cms.execute(command);
            
            if(modifiedContent != null)
            {
               file.getContent().setBytes(modifiedContent);
            }

            //now publish this
            file.getContent().setApprovalProcessId(null);
            file.getContent().setWaitingForPublishApproval(false);
            command = cms.getCommandFactory().createUpdateFileCommand(file, file.getContent(), true);
            cms.execute(command);
         }


         JCRCMS.turnOnWorkflow();
         ACLInterceptor.turnOn();
      }
      else
      {
         ACLInterceptor.turnOff();
         JCRCMS.turnOffWorkflow();
         
         //Check if there are any published versions of this file. If not, this is a new file that was denied publishing
         //hence delete the file itself
         Command command = cms.getCommandFactory().createContentGetVersionsCommand(content.getPath());
         List versions = (List)cms.execute(command);
         List filtered = this.filterPending(versions);
         if(filtered == null || filtered.isEmpty())
         {
            //Delete this file
            Command deleteCommand = cms.getCommandFactory().createDeleteCommand(path);
            cms.execute(deleteCommand);
         }
         
         
         JCRCMS.turnOnWorkflow();
         ACLInterceptor.turnOn();
      }
   }

   /**
    * @param processId
    * @param versions
    * @return
    */
   private org.jboss.portal.cms.model.Content findProcessContent(long processId, List versions)
   {
      org.jboss.portal.cms.model.Content content = null;

      if (versions != null)
      {
         for (int i = 0; i < versions.size(); i++)
         {
            org.jboss.portal.cms.model.Content cour = (org.jboss.portal.cms.model.Content)versions.get(i);
            if (cour.isWaitingForPublishApproval())
            {
               if (cour.getApprovalProcessId() != null &&
                  cour.getApprovalProcessId().trim().equals(String.valueOf(processId)))
               {
                  //not if found...the content of this version should be published to go live
                  return cour;
               }
            }
         }
      }

      return content;
   }
   
   /**
    * 
    * @param versions
    * @return
    */
   private List filterPending(List versions)
   {
      List filtered = new ArrayList();
      
      if(versions != null)
      {
         for(int i=0; i<versions.size(); i++)
         {
            org.jboss.portal.cms.model.Content content = (org.jboss.portal.cms.model.Content)versions.get(i);
            if(!content.isWaitingForPublishApproval())
            {
               filtered.add(content);
            }
         }
      }
      
      return filtered;
   }

   /** @return  */
   private CMS getCMSService()
   {
      try
      {
         return CMSWorkflowUtil.getCMSService();
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
   
   private Content readContent(ExecutionContext executionContext)
   {
      try
      {
         Content content = null;
         
         Object object = executionContext.getContextInstance().getVariable("content");
         content = CMSWorkflowUtil.deserializeContent(object);
         
         return content;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }
}
