/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.portal.identity.db;

import org.jboss.portal.identity.ProfileMap;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Collection;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;

/**
 * A mutable map that expose user properties.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 6280 $
 */
public class ProfileMapImpl implements ProfileMap
{

   /** . */
   private HibernateUserImpl user;

   public ProfileMapImpl(HibernateUserImpl user)
   {
      if (user == null)
      {
         throw new IllegalArgumentException();
      }
      this.user = user;
   }

   public boolean isReadOnly(Object key)
   {
      if (key == null)
      {
         return false;
      }
      if (key instanceof String == false)
      {
         throw new ClassCastException("Key must be a string");
      }
      HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)HibernateUserImpl.ACCESSORS.get(key);
      return accessor != null && !accessor.isWritable();
   }

   public int size()
   {
      return HibernateUserImpl.ACCESSORS.size() + user.getDynamic().size();
   }

   public boolean isEmpty()
   {
      return false;
   }

   /** @throws ClassCastException if the key is not an instance of string */
   public boolean containsKey(Object key) throws ClassCastException
   {
      if (key == null)
      {
         return false;
      }
      if (key instanceof String == false)
      {
         throw new ClassCastException("Key must be a string");
      }
      HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)HibernateUserImpl.ACCESSORS.get(key);
      if (accessor != null)
      {
         return true;
      }
      return user.getDynamic().containsKey(key);
   }

   /** @throws ClassCastException if the value is not an instance of string */
   public boolean containsValue(Object value) throws ClassCastException
   {
      if (value == null)
      {
         throw new NullPointerException("Key cannot be null");
      }
      if (value instanceof String == false)
      {
         throw new ClassCastException("Value must be a string");
      }
      for (Iterator i = HibernateUserImpl.ACCESSORS.values().iterator(); i.hasNext();)
      {
         HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)i.next();
         Object value2 = accessor.get(user);
         if (value == value2 || value.equals(value2))
         {
            return true;
         }
      }
      return user.getDynamic().containsValue(value);
   }

   /** @throws ClassCastException if the key is not an instance of string */
   public Object get(Object key)
   {
      if (key == null)
      {
         return null;
      }
      if (key instanceof String == false)
      {
         throw new ClassCastException("Key must be a string");
      }
      HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)HibernateUserImpl.ACCESSORS.get(key);
      if (accessor != null)
      {
         return accessor.get(user);
      }
      return user.getDynamic().get(key);
   }

   /**
    * Put a value in the dynamic map.
    * <p/>
    * If the key is not an instance of string then an IllegalArgumentException is thrown. If the value is mapped to an
    * accessor of a non writable field then an IllegalArgumentException is thrown. If the value is mapped to an accessor
    * of a non nullable field and the field is null then an IllegalArgumentException is thrown.
    *
    * @throws IllegalArgumentException
    */
   public Object put(Object key, Object newValue) throws IllegalArgumentException
   {
      if (key == null)
      {
         return null;
      }
      if (key instanceof String == false)
      {
         throw new ClassCastException("Key is not a String");
      }
      HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)HibernateUserImpl.ACCESSORS.get(key);
      if (accessor != null)
      {
         if (newValue == null && !accessor.isNullable())
         {
            throw new NullPointerException("Key " + key + " is not nullable");
         }
         if (!accessor.isWritable())
         {
            throw new IllegalArgumentException("Key " + key + " is not modifiable");
         }
         else
         {
            Object oldValue = accessor.get(user);
            if (newValue != null)
            {
               accessor.set(user, newValue);
            }
            else
            {
               accessor.set(user, null);
            }
            return oldValue;
         }
      }
      if (newValue instanceof String == false)
      {
         throw new ClassCastException("Dynamic value must be a string");
      }
      return user.getDynamic().put(key, newValue);
   }

   /**
    * Only affect dynamic properties, otherwise it throws an IllegalArgumentException.
    *
    * @throws IllegalArgumentException if the key is a not a dynamic property
    */
   public Object remove(Object key) throws IllegalArgumentException
   {
      if (key instanceof String == false)
      {
         throw new ClassCastException("Key is not a String");
      }
      HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)HibernateUserImpl.ACCESSORS.get(key);
      if (accessor != null && !accessor.isNullable())
      {
         throw new IllegalArgumentException("Key " + key + " is not removable");
      }
      return user.getDynamic().remove(key);
   }

   /** Clear only dynamic properties. */
   public void clear()
   {
      user.getDynamic().clear();
   }

   public Set keySet()
   {
      // Get
      Set set = new HashSet(size());

      //
      set.addAll(user.getDynamic().keySet());
      set.addAll(HibernateUserImpl.ACCESSORS.keySet());
      return set;
   }

   public Collection values()
   {
      ArrayList collection = new ArrayList(size());
      for (Iterator i = HibernateUserImpl.ACCESSORS.values().iterator(); i.hasNext();)
      {
         HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)i.next();
         collection.add(accessor.get(user));
      }
      collection.addAll(user.getDynamic().values());
      return collection;
   }

   /** Returns an immutable collection of entries. */
   public Set entrySet()
   {
      Map copy = new HashMap(user.getDynamic());
      for (Iterator i = HibernateUserImpl.ACCESSORS.values().iterator(); i.hasNext();)
      {
         HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)i.next();
         copy.put(accessor.getPropertyName(), accessor.get(user));
      }
      return Collections.unmodifiableMap(copy).entrySet();
   }

   public void putAll(Map map)
   {
      // todo : check all properties ok with changes before proceeding
      for (Iterator i = map.entrySet().iterator(); i.hasNext();)
      {
         Entry entry = (Entry)i.next();
         Object key = entry.getKey();
         if (key instanceof String)
         {
            Object value = entry.getValue();
            HibernateUserImpl.PropertyAccessor accessor = (HibernateUserImpl.PropertyAccessor)HibernateUserImpl.ACCESSORS.get(key);
            if (accessor != null)
            {
               if (accessor.isWritable())
               {
                  accessor.set(user, (String)value);
               }
               else
               {
                  // julien : Do something better ?
               }
            }
            else
            {
               user.getDynamic().put(key, value);
            }
         }
      }
   }
}
