package org.jboss.reloaded.api;
/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
import org.jboss.bootstrap.api.descriptor.BootstrapDescriptor;
import org.jboss.bootstrap.api.descriptor.ClassLoaderResourceBootstrapDescriptor;
import org.jboss.bootstrap.api.mc.server.MCServer;
import org.jboss.deployers.client.spi.main.MainDeployer;

/**
 * Contains convenience methods for obtaining {@link BootstrapDescriptor}s 
 * used in constructing a runtime equipped with the
 * Virtual Deployment Framework from a plain {@link MCServer}.  Typical
 * usage is:
 * 
 * <code>// Create a server
 * final MCServer mcServer = MCServerFactory.createServer();
 * 
 * // Add the required bootstrap descriptors
 * final List<BootstrapDescriptor> descriptors = mcServer.getConfiguration().getBootstrapDescriptors();
 * descriptors.add(ReloadedDescriptors.getClassLoadingDescriptor());
 * descriptors.add(ReloadedDescriptors.getVdfDescriptor());
 * 
 * // Start
 * mcServer.start();</code>
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 */
public final class ReloadedShrinkWrapDeployerDescriptors
{
   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Name of the resource containing a Microcontainer
    * deployment descriptor capable of launching VDF elements
    * required for {@link MainDeployer} support
    */
   private static String RESOURCE_NAME_VDF = "org.jboss.reloaded_vdf.xml";

   /**
    * Name of the resource containing a Microcontainer
    * deployment descriptor capable of launching VFS-based
    * ClassLoading
    */
   public static String RESOURCE_NAME_CLASSLOADING = "org.jboss.reloaded_classloading.xml";

   //-------------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Obtains a {@link BootstrapDescriptor} containing components necessary 
    * for starting the Virtual Deployment Framework and {@link MainDeployer}.
    * The descriptor will be found as a {@link ClassLoader} resource
    * of name {@link ReloadedShrinkWrapDeployerDescriptors#RESOURCE_NAME_VDF} from the Thread
    * Context ClassLoader
    * @return
    */
   public static BootstrapDescriptor getVdfDescriptor()
   {
      // Return
      return new ClassLoaderResourceBootstrapDescriptor(RESOURCE_NAME_VDF);
   }

   /**
    * Obtains a {@link BootstrapDescriptor} containing components necessary 
    * for starting the Virtual Deployment Framework and {@link MainDeployer}.
    * The descriptor will be found as a {@link ClassLoader} resource
    * of name {@link ReloadedShrinkWrapDeployerDescriptors#RESOURCE_NAME_VDF} from the specified
    * @return
    * @param cl {@link ClassLoader} used to obtain the resource
    * @throws IllegalArgumentException If the {@link ClassLoader} is not specified
    */
   public static BootstrapDescriptor getVdfDescriptor(final ClassLoader cl)
   {
      // Precondition check
      if (cl == null)
      {
         throw new IllegalArgumentException("ClassLoader must be specified");
      }

      // Return
      return new ClassLoaderResourceBootstrapDescriptor(RESOURCE_NAME_VDF, cl);
   }

   /**
    * Obtains a {@link BootstrapDescriptor} containing components necessary 
    * for starting VFS-based ClassLoading Services
    * @return
    */
   public static BootstrapDescriptor getClassLoadingDescriptor()
   {
      // Return
      return new ClassLoaderResourceBootstrapDescriptor(RESOURCE_NAME_CLASSLOADING);
   }

   /**
    * Obtains a {@link BootstrapDescriptor} containing components necessary 
    * for starting VFS-based ClassLoading Services.
    * The descriptor will be found as a {@link ClassLoader} resource
    * of name {@link ReloadedShrinkWrapDeployerDescriptors#RESOURCE_NAME_CLASSLOADING} from the specified
    * @return
    * @param cl {@link ClassLoader} used to obtain the resource
    * @throws IllegalArgumentException If the {@link ClassLoader} is not specified
    */
   public static BootstrapDescriptor getClassLoadingDescriptor(final ClassLoader cl)
   {
      // Precondition check
      if (cl == null)
      {
         throw new IllegalArgumentException("ClassLoader must be specified");
      }

      // Return
      return new ClassLoaderResourceBootstrapDescriptor(RESOURCE_NAME_CLASSLOADING, cl);
   }

}
