/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.deployers;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.structure.spi.DeploymentUnitFilter;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.scanning.plugins.DeploymentScanningPluginFactory;
import org.jboss.scanning.plugins.DeploymentUnitScanner;
import org.jboss.scanning.spi.ScanningPlugin;
import org.jboss.scanning.spi.helpers.UrlScanner;
import org.jboss.scanning.spi.metadata.ScanningMetaData;

/**
 * Scanning deployer.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class ScanningDeployer extends AbstractDeployer
{
   /** The scanning plugins attachment key */
   public static String PLUGINS_KEY = "SCANNING_PLUGINS_KEY";
   /** The filter */
   private DeploymentUnitFilter filter;
   /** The plugin factories */
   private Set<DeploymentScanningPluginFactory> factories = new CopyOnWriteArraySet<DeploymentScanningPluginFactory>();

   public ScanningDeployer()
   {
      setStage(DeploymentStages.POST_CLASSLOADER);
      addInput(ScanningMetaData.class);
   }

   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      // running this deployer is costly, check if it should be run
      if (filter != null && filter.accepts(unit) == false)
         return;

      Set<ScanningPlugin> plugins = new HashSet<ScanningPlugin>();
      for (DeploymentScanningPluginFactory factory : factories)
      {
         if (factory.isRelevant(unit))
            plugins.add(factory.create(unit));
      }

      if (plugins.isEmpty() == false)
      {
         VFSDeploymentUnit vdu = null;
         if (unit instanceof VFSDeploymentUnit)
            vdu = VFSDeploymentUnit.class.cast(unit);

         try
         {
            UrlScanner scanner = (vdu != null) ? new VFSDeploymentUnitScanner(vdu) : new DeploymentUnitScanner(unit);
            if (scanner.doScan())
            {
               scanner.setPlugins(plugins);
               // check ignore flags
               scanner.setIgnoreIndexedHandles(unit.isAttachmentPresent(UrlScanner.IGNORE_PRE_INDEXED));
               scanner.setIgnoreRuntimeScanning(unit.isAttachmentPresent(UrlScanner.IGNORE_RUNTIME_SCAN));
               // the scan
               scanner.scan();
               // add used plugins as attachment
               unit.addAttachment(PLUGINS_KEY, plugins); // add used plugins as attachment
            }
         }
         catch (Exception e)
         {
            throw DeploymentException.rethrowAsDeploymentException("Error scanning deployment: " + unit, e);
         }
      }
   }

   /**
    * Add factory.
    *
    * @param factory the factory
    */
   public void addFactory(DeploymentScanningPluginFactory factory)
   {
      if (factory == null)
         throw new IllegalArgumentException("Null factory");

      factories.add(factory);
   }

   /**
    * Remove the factory.
    *
    * @param factory the factory
    */
   public void removeFactory(DeploymentScanningPluginFactory factory)
   {
      if (factory == null)
         return;

      factories.remove(factory);
   }

   /**
    * Set vfs deployment filter.
    *
    * @param filter the vfs deployment filter.
    */
   public void setFilter(DeploymentUnitFilter filter)
   {
      this.filter = filter;
   }
}
