/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.helpers;

import java.io.File;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.classloading.spi.visitor.ResourceContext;

/**
 * Cache url owner lookup.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class CachingResourceOwnerFinder implements ResourceOwnerFinder
{
   /** The file to url cache */
   private Map<File, URL> cache = new HashMap<File, URL>();

   /**
    * Cleanup.
    */
   public void cleanup()
   {
      cache.clear();
   }

   /**
    * Get owner/jar url -- w/o actually loading class/resource.
    *
    * @param resource the resource context
    * @return the owner/jar url
    */
   public URL findOwnerURL(ResourceContext resource)
   {
      try
      {
         URL url = resource.getUrl();

         URLConnection conn = url.openConnection();
         if (conn instanceof JarURLConnection)
         {
            JarURLConnection jarConn = (JarURLConnection) url.openConnection();
            return jarConn.getJarFileURL();
         }

         File file = new File(url.getPath());
         if (file.exists()) // does the file exist
            return url;
         url = cache.get(file); // is the file cached
         if (url != null)
            return url;

         List<File> files = new ArrayList<File>();
         files.add(file); // add file, so we cache its mapping
         // move to parent, so we don't duplicate previous lookups
         file = file.getParentFile();

         while(file.exists() == false)
         {
            url = cache.get(file);
            if (url != null)
               break;

            files.add(file); // not found, add to cache
            file = file.getParentFile();

            if (file == null)
               throw new IllegalArgumentException("Cannot find physical file: " + resource.getUrl());
         }

         // not found in cache yet
         if (url == null)
         {
            url = file.toURI().toURL(); // I guess this is the jar url?
            files.add(file); // so we map/cache the current file as well
         }

         for (File f : files)
            cache.put(f, url);

         return url;
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
}