/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.helpers;

import java.lang.reflect.Proxy;
import java.util.Map;
import java.util.WeakHashMap;

import org.jboss.classloading.spi.dependency.Module;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.scanning.plugins.visitor.ReflectProvider;

/**
 * Find the util for deployment.
 * Newly created utils are grouped per module.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class DeploymentUtilsFactory
{
   /** The default impls */
   private static Map<Class<?>, UtilFactory<?>> defaults = new WeakHashMap<Class<?>, UtilFactory<?>>();

   static
   {
      addImplementation(ReflectProvider.class, new ReflectProviderUtilFactory());
      addImplementation(ResourceOwnerFinder.class, new ResourceOwnerFinderUtilFactory());
   }

   /**
    * Add the util impl.
    *
    * @param iface the interface
    * @param factory the util factory
    */
   public static <T> void addImplementation(Class<T> iface, UtilFactory<T> factory)
   {
      defaults.put(iface, factory);
   }

   /**
    * Remove the util impl.
    *
    * @param iface the interface
    */
   public static <T> void removeImplementation(Class<T> iface)
   {
      defaults.remove(iface);
   }

   /**
    * Get util.
    *
    * @param unit the deployment unit
    * @param utilType the util type
    * @return util instance
    */
   public static <T> T getUtil(DeploymentUnit unit, Class<T> utilType)
   {
      if (utilType == null)
         throw new IllegalArgumentException("Null util type");

      DeploymentUnit moduleUnit = getModuleUnit(unit);

      T util = moduleUnit.getAttachment(utilType);
      if (util == null)
      {
         UtilFactory factory = defaults.get(utilType);
         if (factory == null)
            throw new IllegalArgumentException("No util factory defined for " + utilType);

         Object instance = factory.create(moduleUnit);
         util = utilType.cast(instance);

         moduleUnit.addAttachment(utilType, util);
      }
      return util;
   }

   /**
    * Get module unit.
    *
    * @param unit the current unit
    * @return unit containing Module, or exception if no such unit exists
    */
   public static DeploymentUnit getModuleUnit(DeploymentUnit unit)
   {
      if (unit == null)
         throw new IllegalArgumentException("Null unit");

      // group util per module
      DeploymentUnit moduleUnit = unit;
      while(moduleUnit != null && moduleUnit.isAttachmentPresent(Module.class) == false)
         moduleUnit = moduleUnit.getParent();

      if (moduleUnit == null)
         throw new IllegalArgumentException("No module in unit: " + unit);

      return moduleUnit;
   }

   /**
    * Wrap util lookup in lazy lookup.
    *
    * @param unit the deployment unit
    * @param utilType the util type
    * @return lazy util proxy
    */
   public static <T> T getLazyUtilProxy(DeploymentUnit unit, Class<T> utilType)
   {
      // null check is in handler
      LazyUtilsProxyHandler<T> handler = new LazyUtilsProxyHandler<T>(unit, utilType);
      Object proxy = Proxy.newProxyInstance(unit.getClassLoader(), new Class[]{utilType}, handler);
      return utilType.cast(proxy);
   }

   /**
    * Get reflect provider.
    *
    * @param unit the depoyment unit
    * @return the provider
    */
   public static ReflectProvider getProvider(DeploymentUnit unit)
   {
      return getUtil(unit, ReflectProvider.class);
   }

   /**
    * Get finder.
    *
    * @param unit the depoyment unit
    * @return the finder
    */
   public static ResourceOwnerFinder getFinder(DeploymentUnit unit)
   {
      return getUtil(unit, ResourceOwnerFinder.class);
   }

   /**
    * Cleanup the util.
    *
    * @param util the util to cleanup
    */
   public static void cleanup(Object util)
   {
      if (util instanceof CachingResourceOwnerFinder)
         CachingResourceOwnerFinder.class.cast(util).cleanup();
   }
}