/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.visitor;

import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceVisitor;
import org.jboss.logging.Logger;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.TypeInfo;

/**
 * Reflect based resource visitor.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public abstract class ReflectResourceVisitor implements ResourceVisitor
{
   protected final Logger log = Logger.getLogger(getClass());

   private ReflectProvider provider;
   protected boolean ignoreError;

   protected ReflectResourceVisitor(ReflectProvider provider)
   {
      if (provider == null)
         throw new IllegalArgumentException("Null provider");
      this.provider = provider;
   }

   /**
    * Get type info.
    *
    * @param resource the resource
    * @return resource's type info
    * @throws Exception for any error
    */
   protected TypeInfo getTypeInfo(ResourceContext resource) throws Throwable
   {
      return provider.getTypeInfo(resource);
   }

   /**
    * Get class info.
    *
    * @param resource the resource
    * @return resource's type info
    * @throws Exception for any error
    */
   protected ClassInfo getClassInfo(ResourceContext resource) throws Throwable
   {
      TypeInfo typeInfo = getTypeInfo(resource);
      if (typeInfo instanceof ClassInfo == false)
         throw new IllegalArgumentException("Can only handle class info: " + typeInfo);

      return ClassInfo.class.cast(typeInfo);
   }

   public void visit(ResourceContext resource)
   {
      if (resource.isClass() == false)
         return;

      try
      {
         doVisit(resource);
      }
      catch (Throwable t)
      {
         if (ignoreError)
            logThrowable(resource, t);
         else
            throw new RuntimeException(t);
      }
   }

   /**
    * Do visit class resource.
    *
    * @param resource the current resource
    * @throws Throwable for any error
    */
   protected void doVisit(ResourceContext resource) throws Throwable
   {
      ClassInfo classInfo = getClassInfo(resource);
      handleClass(resource, classInfo);
   }

   /**
    * Handle class adapter for annotations.
    *
    * @param resource the current resource
    * @param classInfo the class info instance
    * @throws Exception for any annotations lookup problems
    */
   protected void handleClass(ResourceContext resource, ClassInfo classInfo) throws Exception
   {
      classInfo.setAttachment(ResourceContext.class.getName(), resource);
      try
      {
         handleClass(classInfo);
      }
      finally
      {
         classInfo.setAttachment(ResourceContext.class.getName(), null);
      }
   }

   /**
    * Handle class adapter for annotations.
    *
    * @param classInfo the class info instance
    * @throws Exception for any annotations lookup problems
    */
   protected void handleClass(ClassInfo classInfo) throws Exception
   {
   }

   /**
    * Log throwable.
    *
    * @param resource the resource we're visiting
    * @param t        the throwable
    */
   protected void logThrowable(ResourceContext resource, Throwable t)
   {
      t.printStackTrace();
      if (log.isTraceEnabled())
         log.trace("Exception reading resource: " + resource.getResourceName(), t);
   }

   public void setIgnoreError(boolean ignoreError)
   {
      this.ignoreError = ignoreError;
   }
}
