/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.jboss.classloader.spi.filter.ClassFilter;
import org.jboss.classloading.plugins.vfs.VFSResourceVisitor;
import org.jboss.classloading.plugins.visitor.FederatedResourceVisitor;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;

/**
 * Default scanner.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class DefaultScanner extends VFSUrlScanner
{
   /** The class loader */
   private ClassLoader classLoader;
   private Map<ScanningPlugin, ScanningHandle> handles;

   protected VirtualFile[] excludedRoots;
   protected ClassFilter included;
   protected ClassFilter excluded;

   public DefaultScanner(ClassLoader classLoader, URL... roots)
   {
      super(roots);
      this.classLoader = classLoader;
      this.handles = new HashMap<ScanningPlugin, ScanningHandle>();
      setToVFS(true); // by default we force the vfs
   }

   /**
    * Get the handles.
    *
    * @return the handles
    */
   public Map<ScanningPlugin, ScanningHandle> getHandles()
   {
      return handles;
   }

   /**
    * No root == no sense in scanning.
    *
    * @return true if there are some roots to scan
    */
   public boolean doScan()
   {
      return (getRoots().length > 0);
   }

   protected void scan(Set<ScanningPlugin> plugins) throws Exception
   {
      URL[] urls;
      if (isToVFS())
         urls = toVFS(getRoots());
      else
         urls = getRoots();

      VirtualFile[] roots = new VirtualFile[urls.length];
      for (int i = 0; i < roots.length; i++)
      {
         roots[i] = VFS.getChild(urls[i]);
      }

      ScanningPlugin[] array = plugins.toArray(new ScanningPlugin[plugins.size()]);
      FederatedResourceVisitor visitor = new FederatedResourceVisitor(array, array, getRecurseFilters(array));
      ResourceFilter filter = visitor.getFilter();
      ResourceFilter recurseFilter = visitor.getRecurseFilter();

      // something in javassist uses TCL
      ClassLoader tccl = SecurityActions.setContextClassLoader(classLoader);
      try
      {
         VFSResourceVisitor.visit(roots, excludedRoots, included, excluded, classLoader, visitor, filter, recurseFilter, urls);
      }
      finally
      {
         SecurityActions.resetContextClassLoader(tccl);
      }
   }

   protected ScanningHandle createHandle(ScanningPlugin plugin)
   {
      ScanningHandle handle = plugin.createHandle();
      handles.put(plugin, handle); // remember the created handle
      return handle;
   }

   public void setExcludedRoots(VirtualFile[] excludedRoots)
   {
      this.excludedRoots = excludedRoots;
   }

   public void setIncluded(ClassFilter included)
   {
      this.included = included;
   }

   public void setExcluded(ClassFilter excluded)
   {
      this.excluded = excluded;
   }
}
