/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.indexer.core;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.scanning.plugins.DefaultScanner;
import org.jboss.scanning.spi.Scanner;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;
import org.jboss.vfs.util.automount.Automounter;

/**
 * Scan utils.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 * @author <a href="mailto:jesper.pedersen@jboss.org">Jesper Pedersen</a>
 */
public class ScanUtils
{
   /**
    * Scan the input with given classpath.
    * The input is added as part of classpath,
    * hence no need to explicitly include it.
    *
    * @param input the input jar
    * @param pluginProviders the providers for plugins used while scanning
    * @param cp the classpath
    * @return output file
    * @throws Exception for any error
    */
   public static File scan(File input, Set<String> pluginProviders, URL... cp) throws Exception
   {
      if (input == null)
         throw new IllegalArgumentException("Null input");
      if (pluginProviders == null || pluginProviders.isEmpty())
         throw new IllegalArgumentException("Null or empty providers");
      if (cp == null)
         cp = new URL[0];

      URL root = input.toURI().toURL();
      URL[] urls = new URL[cp.length + 1];
      urls[0] = root;
      System.arraycopy(cp, 0, urls, 1, cp.length);

      URLClassLoader ucl = new URLClassLoader(urls, ScanUtils.class.getClassLoader());
      Set<ScanningPlugin> plugins = new HashSet<ScanningPlugin>();
      for (String provider : pluginProviders)
         plugins.add(createPlugin(provider, ucl));

      DefaultScanner scaner = new DefaultScanner(ucl, root);
      scaner.setPlugins(plugins);
      scaner.setIgnoreIndexedHandles(true); // re-index

      VirtualFile vfs = VFS.getChild(root);
      Automounter.mount(vfs);
      try
      {
         scaner.scan(); // do the actual scanning
      }
      finally
      {
         Automounter.cleanup(vfs);
      }

      // write handles to file
      Map<ScanningPlugin, ScanningHandle> handles = scaner.getHandles();
      File tmp = new File(System.getProperty("java.io.tmpdir"));
      File jar = FileUtil.extract(input, tmp);
      try
      {
         File destination = new File(jar, Scanner.META_INF);
         store(destination, handles);
         File output = new File(input.getParent(), input.getName() + Scanner.SUFFIX);
         FileUtil.compress(jar, output);
         return output;
      }
      finally
      {
         FileUtil.recursiveDelete(jar);
      }
   }

   /**
    * Create plugin.
    *
    * If the provider is actual plugin,
    * it must take classloader as a single parameter with constructor.
    *
    * @param provider the provider; can be actual plugin or intermediate PluginProvider
    * @param ucl the current CL
    * @return new plugin instance
    * @throws Exception for any error
    */
   protected static ScanningPlugin createPlugin(String provider, URLClassLoader ucl) throws Exception
   {
      Class<?> clazz = ucl.loadClass(provider);
      if (ScanningPlugin.class.isAssignableFrom(clazz))
      {
         try
         {
            // let's first try no-arg ctor
            return (ScanningPlugin) clazz.newInstance();
         }
         catch (Exception e)
         {
            // fall back to CL arg ctor
            Constructor<?> ctor = clazz.getDeclaredConstructor(ClassLoader.class);
            return (ScanningPlugin) ctor.newInstance(ucl);
         }
      }
      else if (PluginProvider.class.isAssignableFrom(clazz))
      {
         PluginProvider pp = (PluginProvider) clazz.newInstance();
         return pp.create(ucl);
      }
      else
         throw new IllegalArgumentException("Cannot create plugin from " + provider);
   }

   /**
    * Store attachments.
    *
    * @param directory The destination directory
    * @param handles the handles we scanned
    * @exception Exception Thrown if an error occurs
    */
   protected static void store(File directory, Map<ScanningPlugin, ScanningHandle> handles) throws Exception
   {
      if (handles == null)
         throw new IllegalArgumentException("Null handles");

      if (directory == null)
         throw new IllegalArgumentException("Directory is null");
      if (directory.exists() == false && directory.mkdir() == false)
         throw new IllegalArgumentException("Cannot create directory: " + directory); 

      for (Map.Entry<ScanningPlugin, ScanningHandle> entry : handles.entrySet())
      {
         ScanningPlugin plugin = entry.getKey();
         File file = new File(directory, plugin.getFileName());
         FileOutputStream fos = new FileOutputStream(file);
         BufferedOutputStream bos = new BufferedOutputStream(fos);
         //noinspection unchecked
         plugin.writeHandle(bos, entry.getValue());
      }
   }
}
