/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.hibernate;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.*;

import org.jboss.classloading.plugins.vfs.PackageVisitor;
import org.jboss.classloading.plugins.visitor.FederatedResourceVisitor;
import org.jboss.classloading.spi.dependency.Module;
import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceVisitor;
import org.jboss.deployers.spi.deployer.helpers.AttachmentLocator;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.scanning.plugins.helpers.DeploymentUtilsFactory;
import org.jboss.scanning.plugins.helpers.MergeUtils;
import org.jboss.scanning.plugins.helpers.ResourceOwnerFinder;
import org.jboss.scanning.plugins.helpers.WeakClassLoaderHolder;
import org.jboss.scanning.plugins.visitor.ErrorHandler;
import org.jboss.scanning.plugins.visitor.ReflectProvider;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;

import org.hibernate.AssertionFailure;
import org.hibernate.ejb.packaging.NamedInputStream;
import org.hibernate.ejb.packaging.Scanner;

/**
 * Hibernate's scanner impl.
 *
 * TODO -- should we fix path mapping?
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class ScannerImpl extends WeakClassLoaderHolder implements Scanner, ScanningHandle<ScannerImpl>
{
   /** The packages cache */
   private Map<String, Set<Package>> packages = new HashMap<String, Set<Package>>();
   /** The packages cache */
   private Map<String, Set<String>> pckgCache = new HashMap<String, Set<String>>();
   /** The packages blacklist */
   private Map<String, Set<String>> pckgBlacklist = new HashMap<String, Set<String>>();
   /** The classes cache */
   private Map<String, Map<Class<? extends Annotation>, Set<String>>> classes = new HashMap<String, Map<Class<? extends Annotation>, Set<String>>>();
   /** The files cache */
   private Map<String, Map<String, Set<NamedInputStream>>> files = new HashMap<String, Map<String, Set<NamedInputStream>>>();

   /** The deployment unit */
   private DeploymentUnit unit;
   /** The resource finder */
   private ResourceOwnerFinder finder;
   /** The error handler */
   private ErrorHandler handler;
   /** Do we allow query invocation search */
   private boolean allowQueryInvocationSearch; // by default false, as we expect things to be pre-indexed
   /** Do we cache new results */
   private boolean cacheNewResults;

   private static DeploymentUnit check(DeploymentUnit unit)
   {
      if (unit == null)
         throw new IllegalArgumentException("Null unit");
      return unit;
   }

   public ScannerImpl(DeploymentUnit unit)
   {
      super(check(unit).getClassLoader());
      this.unit = unit;
      this.finder = DeploymentUtilsFactory.getFinder(unit);
      this.handler = DeploymentUtilsFactory.getHandler(unit);

      // allow runtime scanning, but don't cache
      allowQueryInvocationSearch = DeploymentUtilsFactory.getAttachment(unit, "org.jboss.scanning.hibernate.allowQueryInvocationSearch", Boolean.class, Boolean.TRUE);      
      cacheNewResults = DeploymentUtilsFactory.getAttachment(unit, "org.jboss.scanning.hibernate.cacheNewResults", Boolean.class, Boolean.FALSE);
   }

   /**
    * Cleanup.
    */
   protected void cleanup()
   {
      packages.clear();
      pckgCache.clear();
      pckgBlacklist.clear();
      classes.clear();
      files.clear();
   }

   /**
    * Get owner/jar url -- w/o actually loading class/resource.
    *
    * @param resource the resource context
    * @return the owner/jar url
    */
   URL getOwnerURL(ResourceContext resource)
   {
      return finder.findOwnerURL(resource);
   }

   ErrorHandler getErrorHandler()
   {
      return handler;
   }

   Package loadPackage(String pckg) throws ClassNotFoundException
   {
      return loadClassWithCNFE(pckg + ".package-info").getPackage();
   }

   void addPackage(URL url, String pckg)
   {
      String path = url.getPath();

      // blacklist
      Set<String> blacklist = pckgBlacklist.get(path);
      if (blacklist != null && blacklist.contains(pckg))
         return;

      // cache
      Set<String> cache = pckgCache.get(path);
      if (cache != null)
      {
         if (cache.contains(pckg))
            return;
      }
      else
      {
         cache = new HashSet<String>();
         pckgCache.put(path, cache);
      }

      cache.add(pckg); // cache result

      Set<Package> pckgs = packages.get(path);
      if (pckgs == null)
      {
         pckgs = new HashSet<Package>();
         packages.put(path, pckgs);
      }
      try
      {
         Package aPackage = loadPackage(pckg);
         pckgs.add(aPackage);
      }
      catch (ClassNotFoundException e)
      {
         if (blacklist == null)
         {
            blacklist = new HashSet<String>();
            pckgBlacklist.put(path, blacklist);
         }
         blacklist.add(pckg);
      }
   }

   void addClass(URL url, Class<? extends Annotation> annotation, String clazz)
   {
      String path = url.getPath();
      Map<Class<? extends Annotation>, Set<String>> map = classes.get(path);
      if (map == null)
      {
         map = new HashMap<Class<? extends Annotation>, Set<String>>();
         classes.put(path, map);
      }
      Set<String> cls = map.get(annotation);
      if (cls == null)
      {
         cls = new HashSet<String>();
         map.put(annotation, cls);
      }
      cls.add(clazz);
   }

   void addFile(URL url, String pattern, NamedInputStream nis)
   {
      String path = url.getPath();
      Map<String, Set<NamedInputStream>> map = files.get(path);
      if (map == null)
      {
         map = new HashMap<String, Set<NamedInputStream>>();
         files.put(path, map);
      }
      Set<NamedInputStream> nims = map.get(pattern);
      if (nims == null)
      {
         nims = new HashSet<NamedInputStream>();
         map.put(pattern, nims);
      }
      nims.add(nis);
   }

   public void merge(ScannerImpl subHandle)
   {
      MergeUtils.singleMerge(packages, subHandle.getPackages());
      MergeUtils.doubleMerge(classes, subHandle.getClasses());
      MergeUtils.doubleMerge(files, subHandle.getFiles());
   }

   public Set<Package> getPackagesInJar(URL jarToScan, Set<Class<? extends Annotation>> annotationsToLookFor)
   {
      if (jarToScan == null)
         throw new IllegalArgumentException("Null jar to scan url");
      if (annotationsToLookFor == null)
         throw new IllegalArgumentException("Null annotations to look for");

      if (annotationsToLookFor.size() > 0)
         throw new AssertionFailure("Improper use of MC Scanner: must not filter packages");

      String path = jarToScan.getPath();
      Set<Package> p = packages.get(path);
      if (p == null && allowQueryInvocationSearch)
      {
         Module module = AttachmentLocator.searchAncestors(unit, Module.class);
         if (module == null)
            throw new IllegalArgumentException("No such module: " + unit);

         p = new HashSet<Package>();

         VirtualFile root = getFile(jarToScan);
         Set<String> pckgs = PackageVisitor.determineAllPackages(
               new VirtualFile[]{root},
               null,
               module.getExportAll(),
               module.getIncluded(),
               module.getExcluded(),
               module.getExcludedExport());
         for (String pckg : pckgs)
         {
            try
            {
               Package pck = loadPackage(pckg);
               p.add(pck);
            }
            catch (ClassNotFoundException ignore)
            {
            }
         }

         if (cacheNewResults)
            packages.put(path, p);
      }

      // remove cached string packages
      pckgCache.remove(path);

      return p != null ? Collections.unmodifiableSet(p) : Collections.<Package>emptySet();
   }

   public Set<Class<?>> getClassesInJar(URL jartoScan, Set<Class<? extends Annotation>> annotationsToLookFor)
   {
      if (jartoScan == null)
         throw new IllegalArgumentException("Null jar to scan url");
      if (annotationsToLookFor == null)
         throw new IllegalArgumentException("Null annotations to look for");

      Set<String> strings = new HashSet<String>();
      if (annotationsToLookFor.isEmpty())
      {
         Module module = AttachmentLocator.searchAncestors(unit, Module.class);
         if (module == null)
            throw new IllegalArgumentException("No such module: " + unit);

         ClassesVisitor visitor = new ClassesVisitor(strings);
         module.visit(visitor, visitor.getFilter(), null, jartoScan);
      }
      else
      {
         Set<Class<? extends Annotation>> missingCacheAnnotations = new HashSet<Class<? extends Annotation>>();
         Map<Class<? extends Annotation>, Set<String>> map = classes.get(jartoScan.getPath());
         if (map != null)
         {
            for (Class<? extends Annotation> annClass : annotationsToLookFor)
            {
               Set<String> s = map.get(annClass);
               if (s != null)
               {
                  strings.addAll(s);
               }
               else
               {
                  missingCacheAnnotations.add(annClass);
               }
            }
         }
         else
         {
            missingCacheAnnotations.addAll(annotationsToLookFor);
         }

         if (allowQueryInvocationSearch && missingCacheAnnotations.isEmpty() == false)
         {
            Module module = AttachmentLocator.searchAncestors(unit, Module.class);
            if (module == null)
               throw new IllegalArgumentException("No such module: " + unit);

            Map<Class<? extends Annotation>, Set<String>> temp = new HashMap<Class<? extends Annotation>, Set<String>>();
            ResourceVisitor[] visitors = new ResourceVisitor[missingCacheAnnotations.size()];
            ReflectProvider provider = DeploymentUtilsFactory.getProvider(unit);
            int i = 0;
            for (Class<? extends Annotation> annotation : missingCacheAnnotations)
            {
               Set<String> tmpStrings = new HashSet<String>();
               temp.put(annotation, tmpStrings);
               TempAnnotationVisitor tav = new TempAnnotationVisitor(provider, annotation, strings);
               tav.setErrorHandler(getErrorHandler());
               visitors[i++] = tav;
            }
            ResourceVisitor visitor = new FederatedResourceVisitor(visitors, null, null);
            module.visit(visitor, visitor.getFilter(), null, jartoScan);

            if (cacheNewResults)
            {
               if (map != null)
                  MergeUtils.singleMerge(map, temp);
               else
                  classes.put(jartoScan.getPath(), temp);
            }
            for (Set<String> vals : temp.values())
               strings.addAll(vals);
         }
      }

      Set<Class<?>> result = new HashSet<Class<?>>();
      for (String clazz : strings)
         result.add(loadClass(clazz));
      return result;
   }

   public Set<NamedInputStream> getFilesInJar(URL jartoScan, Set<String> filePatterns)
   {
      if (jartoScan == null)
         throw new IllegalArgumentException("Null jar to scan url");
      if (filePatterns == null)
         throw new IllegalArgumentException("Null file patterns to look for");

      Set<NamedInputStream> result = new HashSet<NamedInputStream>();
      String path = jartoScan.getPath();
      Map<String, Set<NamedInputStream>> map = files.get(path);
      if (map != null)
      {
         findFiles(jartoScan, filePatterns, map, result);
      }
      else
      {
         map = new HashMap<String, Set<NamedInputStream>>();
         findFiles(jartoScan, filePatterns, map, result);
         if (cacheNewResults)
            files.put(path, map);
      }
      return result;
   }

   private void findFiles(URL jartoScan, Set<String> filePatterns, Map<String, Set<NamedInputStream>> map, Set<NamedInputStream> result)
   {
      if (filePatterns.isEmpty())
      {
         for (Set<NamedInputStream> nims : map.values())
            result.addAll(nims);
      }
      else
      {
         VirtualFile root = null;
         for (String pattern : filePatterns)
         {
            Set<NamedInputStream> niss = map.get(pattern);
            if (niss == null && allowQueryInvocationSearch)
            {
               if (root == null)
                  root = getFile(jartoScan);

               try
               {
                  List<VirtualFile> children = root.getChildrenRecursively(new PatternFilter(pattern));
                  niss = toNIS(children);
                  if (cacheNewResults)
                     map.put(pattern, niss);
               }
               catch (IOException e)
               {
                  throw new RuntimeException(e);
               }
            }
            if (niss != null)
               result.addAll(niss);
         }
      }
   }

   private Set<NamedInputStream> toNIS(Iterable<VirtualFile> files)
   {
      Set<NamedInputStream> result = new HashSet<NamedInputStream>();
      for (VirtualFile file : files)
      {
         NamedInputStream nis = new VirtualFileNamedInputStream(file);
         result.add(nis);
      }
      return result;
   }

   // For back compatible Hibernate versions
   @SuppressWarnings("unused")
   public Set<NamedInputStream> getFilesInClasspath(URL jartoScan, Set<String> filePatterns)
   {
      throw new RuntimeException("Not yet implemented");
   }

   public Set<NamedInputStream> getFilesInClasspath(Set<String> filePatterns)
   {
      throw new RuntimeException("Not yet implemented");
   }

   public String getUnqualifiedJarName(URL jarUrl)
   {
      if (jarUrl == null)
         throw new IllegalArgumentException("Null jar url");


      VirtualFile file = getFile(jarUrl);
      return file.getName();
   }

   private VirtualFile getFile(URL url)
   {
      try
      {
         return VFS.getChild(url);
      }
      catch (URISyntaxException e)
      {
         throw new IllegalArgumentException(e);
      }
   }

   public void setAllowQueryInvocationSearch(boolean allowQueryInvocationSearch)
   {
      this.allowQueryInvocationSearch = allowQueryInvocationSearch;
   }

   public void setCacheNewResults(boolean cacheNewResults)
   {
      this.cacheNewResults = cacheNewResults;
   }

   Map<String, Set<Package>> getPackages()
   {
      return packages;
   }

   Map<String, Map<Class<? extends Annotation>, Set<String>>> getClasses()
   {
      return classes;
   }

   Map<String, Map<String, Set<NamedInputStream>>> getFiles()
   {
      return files;
   }
}
