/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.spi;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.classloading.spi.visitor.ResourceVisitor;

/**
 * Scanning plugin.
 * Defines what to do with a resource.
 *
 * @param <T> exact handle type
 * @param <U> exact handle interface
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public interface ScanningPlugin<T extends ScanningHandle, U> extends ResourceFilter, ResourceVisitor
{
   /**
    * Create plugins handle/utility.
    * e.g. AnnotationRepository for annotations scanning
    *
    * @return new handle instance
    */
   T createHandle();

   /**
    * Read serialized handle.
    *
    * @param is the serialized handle's input stream.
    * @return de-serialized handle
    * @throws Exception for any error
    */
   ScanningHandle readHandle(InputStream is) throws Exception;

   /**
    * Write / serialize handle.
    *
    * @param os the output stream to serialize handle.
    * @param handle the handle
    * @throws IOException for any IO error
    */
   void writeHandle(OutputStream os, T handle) throws IOException;

   /**
    * Cleanup handle.
    *
    * @param handle the handle to cleanup
    */
   void cleanupHandle(U handle);

   /**
    * Get handle interface.
    *
    * @return the handle interface
    */
   Class<U> getHandleInterface();

   /**
    * Get handle's key.
    * Used to attach handle to map/attachments.
    *
    * @return the handle's key
    */
   String getAttachmentKey();

   /**
    * Get handle's file name.
    * Used to attach handle to jar and/or get pre-indexed.
    *
    * @return the handle's file name
    */
   String getFileName();

   /***
    * Get recurse filter.
    *
    * @return the recurse filter
    */
   ResourceFilter getRecurseFilter();
}