/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.spi.helpers;

import java.io.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.scanning.spi.Scanner;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;

/**
 * Abstract scanning plugin wrapper
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public abstract class AbstractScanningPlugin<T extends ScanningHandle, U> implements ScanningPlugin<T, U>
{
   public T createHandle()
   {
      T handle = doCreateHandle();
      if (getHandleInterface().isInstance(handle) == false)
         throw new IllegalArgumentException("Illegal handle type: " + handle + " != " + getHandleInterface());
      return handle;
   }

   protected abstract T doCreateHandle();

   protected ObjectInputStream createObjectInputStream(InputStream is) throws IOException
   {
      return new ObjectInputStream(is);
   }

   public ScanningHandle readHandle(InputStream is) throws Exception
   {
      try
      {
         GZIPInputStream gis = new GZIPInputStream(is);
         ObjectInputStream ois = createObjectInputStream(gis);
         return (ScanningHandle) ois.readObject();
      }
      finally
      {
         is.close();
      }
   }

   public void writeHandle(OutputStream os, T handle) throws IOException
   {
      GZIPOutputStream gos = new GZIPOutputStream(os);
      ObjectOutputStream oos = new ObjectOutputStream(gos);
      try
      {
         oos.writeObject(handle);
         oos.flush();
      }
      finally
      {
         oos.close();
      }
   }

   public void cleanupHandle(U handle)
   {
      // by default, do nothing
   }

   public String getAttachmentKey()
   {
      return getHandleInterface().getName();
   }

   public String getFileName()
   {
      return getHandleInterface().getSimpleName() + Scanner.SUFFIX;
   }

   public boolean accepts(ResourceContext resource)
   {
      return getFilter().accepts(resource);
   }

   public ResourceFilter getRecurseFilter()
   {
      return null;
   }
}