/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.spi.helpers;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.List;

import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;
import org.jboss.util.collection.CollectionsFactory;

/**
 * Scanning plugin wrapper
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class ScanningPluginWrapper<T extends ScanningHandle, U> implements ScanningPlugin<T, U>
{
   private ScanningPlugin<T, U> delegate;
   private List<String> excudedPaths = CollectionsFactory.createLazyList();

   public ScanningPluginWrapper(ScanningPlugin<T, U> delegate)
   {
      this.delegate = delegate;
   }

   void addURL(URL url)
   {
      excudedPaths.add(url.getPath());
   }

   public T createHandle()
   {
      return delegate.createHandle();
   }

   public ScanningHandle readHandle(InputStream is) throws Exception
   {
      return delegate.readHandle(is);
   }

   public void writeHandle(OutputStream os, T handle) throws IOException
   {
      delegate.writeHandle(os, handle);
   }

   public void cleanupHandle(U handle)
   {
      delegate.cleanupHandle(handle);
   }

   public Class<U> getHandleInterface()
   {
      return delegate.getHandleInterface();
   }

   public String getAttachmentKey()
   {
      return delegate.getAttachmentKey();
   }

   public String getFileName()
   {
      return delegate.getFileName();
   }

   public ResourceFilter getRecurseFilter()
   {
      if (excudedPaths.isEmpty())
         return delegate.getRecurseFilter();
      else
         return new ExcludedPathsFilter();
   }

   public boolean accepts(ResourceContext resource)
   {
      return delegate.accepts(resource);
   }

   public ResourceFilter getFilter()
   {
      return delegate.getFilter();
   }

   public void visit(ResourceContext resource)
   {
      delegate.visit(resource);
   }

   /**
    * Do not recurse into paths that already have pre-indexed handle.
    */
   private class ExcludedPathsFilter implements ResourceFilter
   {
      public boolean accepts(ResourceContext resource)
      {
         URL url = resource.getUrl();
         String urlString = url.toString();
         for (String exceluded : excudedPaths)
         {
            if (urlString.contains(exceluded))
               return false;
         }
         return delegate.accepts(resource);
      }
   }
}