/*     */ package org.jboss.messaging.core.impl;
/*     */ 
/*     */ import EDU.oswego.cs.dl.util.concurrent.SynchronizedInt;
/*     */ import java.util.ArrayList;
/*     */ import java.util.HashSet;
/*     */ import java.util.Iterator;
/*     */ import java.util.List;
/*     */ import java.util.ListIterator;
/*     */ import java.util.Set;
/*     */ import org.jboss.jms.server.MessagingTimeoutFactory;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.messaging.core.contract.Channel;
/*     */ import org.jboss.messaging.core.contract.Delivery;
/*     */ import org.jboss.messaging.core.contract.DeliveryObserver;
/*     */ import org.jboss.messaging.core.contract.Distributor;
/*     */ import org.jboss.messaging.core.contract.Filter;
/*     */ import org.jboss.messaging.core.contract.Message;
/*     */ import org.jboss.messaging.core.contract.MessageReference;
/*     */ import org.jboss.messaging.core.contract.PersistenceManager;
/*     */ import org.jboss.messaging.core.impl.tx.Transaction;
/*     */ import org.jboss.messaging.core.impl.tx.TransactionException;
/*     */ import org.jboss.messaging.core.impl.tx.TxCallback;
/*     */ import org.jboss.messaging.util.prioritylinkedlist.BasicPriorityLinkedList;
/*     */ import org.jboss.messaging.util.prioritylinkedlist.PriorityLinkedList;
/*     */ import org.jboss.util.timeout.Timeout;
/*     */ import org.jboss.util.timeout.TimeoutFactory;
/*     */ import org.jboss.util.timeout.TimeoutTarget;
/*     */ 
/*     */ public abstract class ChannelSupport
/*     */   implements Channel
/*     */ {
/*  66 */   private static final Logger log = Logger.getLogger(ChannelSupport.class);
/*     */ 
/*  72 */   private boolean trace = log.isTraceEnabled();
/*     */   protected long channelID;
/*     */   protected Distributor distributor;
/*     */   protected boolean receiversReady;
/*     */   protected PriorityLinkedList messageRefs;
/*     */   protected boolean recoverable;
/*     */   protected PersistenceManager pm;
/*     */   protected Object lock;
/*     */   protected volatile boolean active;
/*     */   protected SynchronizedInt deliveringCount;
/*     */   protected Set scheduledDeliveries;
/*     */   protected int maxSize;
/*     */   protected SynchronizedInt messagesAdded;
/*     */ 
/*     */   protected ChannelSupport(long channelID, PersistenceManager pm, boolean recoverable, int maxSize)
/*     */   {
/* 110 */     if (this.trace) log.trace("creating " + (pm != null ? "recoverable " : "non-recoverable ") + "channel[" + channelID + "]");
/*     */ 
/* 112 */     this.pm = pm;
/*     */ 
/* 114 */     this.channelID = channelID;
/*     */ 
/* 116 */     this.recoverable = recoverable;
/*     */ 
/* 118 */     this.messageRefs = new BasicPriorityLinkedList(10);
/*     */ 
/* 120 */     this.lock = new Object();
/*     */ 
/* 122 */     this.deliveringCount = new SynchronizedInt(0);
/*     */ 
/* 124 */     this.scheduledDeliveries = new HashSet();
/*     */ 
/* 126 */     this.maxSize = maxSize;
/*     */ 
/* 128 */     this.messagesAdded = new SynchronizedInt(0);
/*     */   }
/*     */ 
/*     */   public Delivery handle(DeliveryObserver sender, MessageReference ref, Transaction tx)
/*     */   {
/* 135 */     if (!isActive())
/*     */     {
/* 137 */       if (this.trace) log.trace(this + " is not active, returning null delivery for " + ref);
/*     */ 
/* 139 */       return null;
/*     */     }
/*     */ 
/* 142 */     checkClosed();
/*     */ 
/* 144 */     return handleInternal(sender, ref, tx, true);
/*     */   }
/*     */ 
/*     */   public void acknowledge(Delivery d, Transaction tx)
/*     */     throws Throwable
/*     */   {
/* 151 */     if (this.trace) log.trace("acknowledging " + d + (tx == null ? " non-transactionally" : new StringBuilder().append(" transactionally in ").append(tx).toString()));
/*     */ 
/* 153 */     acknowledgeInternal(d, tx, true);
/*     */   }
/*     */ 
/*     */   public void cancel(Delivery del)
/*     */     throws Throwable
/*     */   {
/* 160 */     MessageReference ref = del.getReference();
/*     */ 
/* 162 */     if (ref.getMessage().isReliable())
/*     */     {
/* 164 */       this.pm.updateDeliveryCount(this.channelID, ref);
/*     */     }
/*     */ 
/* 167 */     if (!del.isRecovered())
/*     */     {
/* 169 */       this.deliveringCount.decrement();
/*     */     }
/*     */ 
/* 172 */     if (!checkAndSchedule(ref))
/*     */     {
/* 174 */       cancelInternal(ref);
/*     */     }
/*     */   }
/*     */ 
/*     */   public long getChannelID()
/*     */   {
/* 182 */     return this.channelID;
/*     */   }
/*     */ 
/*     */   public boolean isRecoverable()
/*     */   {
/* 187 */     return this.recoverable;
/*     */   }
/*     */ 
/*     */   public List browse(Filter filter)
/*     */   {
/* 192 */     if (this.trace) log.trace(this + " browse" + (filter == null ? "" : new StringBuilder().append(", filter = ").append(filter).toString()));
/*     */ 
/* 194 */     synchronized (this.lock)
/*     */     {
/* 203 */       List references = undelivered(filter);
/*     */ 
/* 206 */       ArrayList messages = new ArrayList(references.size());
/* 207 */       for (Iterator i = references.iterator(); i.hasNext(); )
/*     */       {
/* 209 */         MessageReference ref = (MessageReference)i.next();
/* 210 */         messages.add(ref.getMessage());
/*     */       }
/* 212 */       return messages;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void deliver()
/*     */   {
/* 218 */     checkClosed();
/*     */ 
/* 220 */     synchronized (this.lock)
/*     */     {
/* 222 */       if ((this.distributor != null) && (this.distributor.getNumberOfReceivers() > 0))
/*     */       {
/* 224 */         setReceiversReady(true);
/*     */ 
/* 226 */         deliverInternal();
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   public void close()
/*     */   {
/* 233 */     synchronized (this.lock)
/*     */     {
/* 235 */       if (this.distributor != null)
/*     */       {
/* 237 */         this.distributor.clear();
/*     */ 
/* 239 */         this.distributor = null;
/*     */       }
/*     */ 
/* 242 */       clearAllScheduledDeliveries();
/*     */     }
/*     */   }
/*     */ 
/*     */   public void removeAllReferences()
/*     */     throws Throwable
/*     */   {
/* 256 */     synchronized (this.lock)
/*     */     {
/* 258 */       if (this.deliveringCount.get() > 0)
/*     */       {
/* 260 */         throw new IllegalStateException("Cannot remove references while deliveries are in progress, there are " + this.deliveringCount.get());
/*     */       }
/*     */ 
/* 264 */       log.trace(this + " removing all references, there are " + this.messageRefs.size());
/*     */       MessageReference ref;
/* 277 */       while ((ref = removeFirstInMemory()) != null)
/*     */       {
/* 279 */         log.trace("Removing ref " + ref);
/*     */ 
/* 281 */         SimpleDelivery del = new SimpleDelivery(this, ref);
/*     */ 
/* 283 */         del.acknowledge(null);
/*     */       }
/*     */ 
/* 286 */       this.deliveringCount.set(0);
/*     */ 
/* 288 */       log.trace(this + " done removing all references, there are " + this.messageRefs.size());
/*     */     }
/*     */ 
/* 291 */     clearAllScheduledDeliveries();
/*     */   }
/*     */ 
/*     */   public List undelivered(Filter filter)
/*     */   {
/* 296 */     List undelivered = new ArrayList();
/*     */ 
/* 298 */     synchronized (this.lock)
/*     */     {
/* 300 */       Iterator iter = this.messageRefs.getAll().iterator();
/*     */ 
/* 302 */       while (iter.hasNext())
/*     */       {
/* 304 */         MessageReference r = (MessageReference)iter.next();
/*     */ 
/* 309 */         if ((filter == null) || (filter.accept(r.getMessage())))
/*     */         {
/* 311 */           undelivered.add(r);
/*     */         }
/* 315 */         else if (this.trace) log.trace(this + ": " + r + " NOT accepted by filter so won't add to list");
/*     */       }
/*     */     }
/*     */ 
/* 319 */     if (this.trace) log.trace(this + ": undelivered() returns a list of " + undelivered.size() + " undelivered memory messages");
/*     */ 
/* 321 */     return undelivered;
/*     */   }
/*     */ 
/*     */   public int getMessageCount()
/*     */   {
/* 329 */     synchronized (this.lock)
/*     */     {
/* 331 */       if (this.trace) log.trace("Getting message count mr: " + this.messageRefs.size() + " dc " + getDeliveringCount() + " sc " + getScheduledCount());
/*     */ 
/* 333 */       return this.messageRefs.size() + getDeliveringCount() + getScheduledCount();
/*     */     }
/*     */   }
/*     */ 
/*     */   public int getDeliveringCount()
/*     */   {
/* 339 */     return this.deliveringCount.get();
/*     */   }
/*     */ 
/*     */   public int getScheduledCount()
/*     */   {
/* 344 */     synchronized (this.scheduledDeliveries)
/*     */     {
/* 346 */       return this.scheduledDeliveries.size();
/*     */     }
/*     */   }
/*     */ 
/*     */   public void activate()
/*     */   {
/* 352 */     this.active = true;
/*     */   }
/*     */ 
/*     */   public void deactivate()
/*     */   {
/* 357 */     this.active = false;
/*     */   }
/*     */ 
/*     */   public boolean isActive()
/*     */   {
/* 362 */     return this.active;
/*     */   }
/*     */ 
/*     */   public int getMaxSize()
/*     */   {
/* 367 */     synchronized (this.lock)
/*     */     {
/* 369 */       return this.maxSize;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void setMaxSize(int newSize)
/*     */   {
/* 375 */     synchronized (this.lock)
/*     */     {
/* 377 */       int count = getMessageCount();
/*     */ 
/* 379 */       if ((newSize != -1) && (count > newSize))
/*     */       {
/* 381 */         log.warn("Cannot set maxSize to " + newSize + " since there are already " + count + " refs");
/*     */       }
/*     */       else
/*     */       {
/* 385 */         this.maxSize = newSize;
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   public int getMessagesAdded()
/*     */   {
/* 392 */     return this.messagesAdded.get();
/*     */   }
/*     */ 
/*     */   public int memoryRefCount()
/*     */   {
/* 400 */     synchronized (this.lock)
/*     */     {
/* 402 */       return this.messageRefs.size();
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void clearAllScheduledDeliveries()
/*     */   {
/* 412 */     synchronized (this.scheduledDeliveries)
/*     */     {
/* 414 */       Set clone = new HashSet(this.scheduledDeliveries);
/*     */ 
/* 416 */       Iterator iter = clone.iterator();
/*     */ 
/* 418 */       while (iter.hasNext())
/*     */       {
/* 420 */         Timeout timeout = (Timeout)iter.next();
/*     */ 
/* 422 */         timeout.cancel();
/*     */       }
/*     */ 
/* 425 */       this.scheduledDeliveries.clear();
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void cancelInternal(MessageReference ref) throws Exception
/*     */   {
/* 431 */     if (this.trace) log.trace(this + " cancelling " + ref + " in memory");
/*     */ 
/* 433 */     synchronized (this.lock)
/*     */     {
/* 435 */       this.messageRefs.addFirst(ref, ref.getMessage().getPriority());
/*     */     }
/*     */ 
/* 438 */     if (this.trace) log.trace(this + " added " + ref + " back into state");
/*     */   }
/*     */ 
/*     */   protected void deliverInternal()
/*     */   {
/* 449 */     if (this.trace) log.trace(this + " was prompted delivery");
/*     */ 
/*     */     try
/*     */     {
/* 455 */       ListIterator iter = null;
/*     */ 
/* 457 */       MessageReference ref = null;
/*     */ 
/* 459 */       if (!getReceiversReady())
/*     */       {
/* 461 */         if (this.trace) log.trace(this + " receivers not ready so not delivering");
/*     */ 
/* 463 */         return;
/*     */       }
/*     */ 
/*     */       while (true)
/*     */       {
/* 468 */         ref = nextReference(iter);
/*     */ 
/* 470 */         if (ref != null)
/*     */         {
/* 474 */           if (this.trace) log.trace(this + " pushing " + ref);
/*     */ 
/* 476 */           Delivery del = this.distributor.handle(this, ref, null);
/*     */ 
/* 478 */           setReceiversReady(del != null);
/*     */ 
/* 480 */           if (del == null)
/*     */           {
/* 483 */             if (!this.trace) break; log.trace(this + " got no delivery for " + ref + " so no receiver got the message. Stopping delivery.");
/*     */           }
/*     */           else
/*     */           {
/* 487 */             if (!del.isSelectorAccepted())
/*     */             {
/* 493 */               if (iter == null)
/*     */               {
/* 495 */                 iter = this.messageRefs.iterator();
/*     */ 
/* 498 */                 iter.next();
/*     */               }
/*     */             }
/*     */             else
/*     */             {
/* 503 */               if (this.trace) log.trace(this + ": " + del + " returned for message " + ref);
/*     */ 
/* 507 */               synchronized (this.lock)
/*     */               {
/* 509 */                 if (iter == null)
/*     */                 {
/* 511 */                   if (this.trace) log.trace(this + " removing first ref in memory");
/*     */ 
/* 513 */                   removeFirstInMemory();
/*     */                 }
/*     */                 else
/*     */                 {
/* 517 */                   if (this.trace) log.trace(this + " removed current message from iterator");
/*     */ 
/* 519 */                   iter.remove();
/*     */                 }
/*     */               }
/*     */ 
/* 523 */               this.deliveringCount.increment();
/*     */             }
/* 525 */             continue;
/*     */           }
/*     */         }
/*     */         else {
/* 529 */           if (!this.trace) break; log.trace(this + " no more refs to deliver ");
/*     */         }
/*     */ 
/*     */       }
/*     */ 
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 537 */       log.error(this + " Failed to deliver", t);
/*     */     }
/*     */   }
/*     */ 
/*     */   protected boolean deliverScheduled(MessageReference ref)
/*     */   {
/*     */     try
/*     */     {
/* 547 */       synchronized (this.lock)
/*     */       {
/* 551 */         if (this.trace) log.trace(this + " pushing " + ref);
/*     */ 
/* 553 */         Delivery del = this.distributor.handle(this, ref, null);
/*     */ 
/* 555 */         setReceiversReady(del != null);
/*     */ 
/* 557 */         if (del == null)
/*     */         {
/* 560 */           if (this.trace) log.trace(this + ": no delivery returned for message" + ref + " so no receiver got the message. Delivery is now complete");
/*     */ 
/* 562 */           return false;
/*     */         }
/* 564 */         if (del.isSelectorAccepted())
/*     */         {
/* 566 */           if (this.trace) log.trace(this + ": " + del + " returned for message:" + ref);
/*     */ 
/* 570 */           this.deliveringCount.increment();
/*     */ 
/* 572 */           return true;
/*     */         }
/*     */       }
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 578 */       log.error(this + " Failed to deliver", t);
/*     */     }
/*     */ 
/* 581 */     return false;
/*     */   }
/*     */ 
/*     */   protected Delivery handleInternal(DeliveryObserver sender, MessageReference ref, Transaction tx, boolean persist)
/*     */   {
/* 587 */     if (ref == null)
/*     */     {
/* 589 */       return null;
/*     */     }
/*     */ 
/* 592 */     if (this.trace) log.trace(this + " handles " + ref + (tx == null ? " non-transactionally" : new StringBuilder().append(" in transaction: ").append(tx).toString()));
/*     */ 
/* 594 */     if ((this.maxSize != -1) && (getMessageCount() >= this.maxSize))
/*     */     {
/* 598 */       log.warn(this + " has reached maximum size, " + ref + " will be dropped");
/*     */ 
/* 600 */       return null;
/*     */     }
/*     */ 
/* 604 */     ref = ref.copy();
/*     */     try
/*     */     {
/* 608 */       if (tx == null)
/*     */       {
/* 610 */         if ((persist) && (ref.getMessage().isReliable()) && (this.recoverable))
/*     */         {
/* 613 */           if (this.trace) log.trace(this + " adding " + ref + " to database non-transactionally");
/*     */ 
/* 616 */           this.pm.addReference(this.channelID, ref, null);
/*     */         }
/*     */ 
/* 622 */         if (!checkAndSchedule(ref))
/*     */         {
/* 624 */           synchronized (this.lock)
/*     */           {
/* 626 */             addReferenceInMemory(ref);
/*     */ 
/* 628 */             deliverInternal();
/*     */           }
/*     */         }
/*     */       }
/*     */       else
/*     */       {
/* 634 */         if (this.trace) log.trace(this + " adding " + ref + " to state " + (tx == null ? "non-transactionally" : new StringBuilder().append("in transaction: ").append(tx).toString()));
/*     */ 
/* 637 */         getCallback(tx).addRef(ref);
/*     */ 
/* 639 */         if (this.trace) log.trace(this + " added transactionally " + ref + " in memory");
/*     */ 
/* 641 */         if ((persist) && (ref.getMessage().isReliable()) && (this.recoverable))
/*     */         {
/* 644 */           if (this.trace) log.trace(this + " adding " + ref + (tx == null ? " to database non-transactionally" : new StringBuilder().append(" in transaction: ").append(tx).toString()));
/*     */ 
/* 646 */           this.pm.addReference(this.channelID, ref, tx);
/*     */         }
/*     */       }
/*     */ 
/* 650 */       this.messagesAdded.increment();
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 654 */       log.error("Failed to handle message", t);
/*     */ 
/* 656 */       return null;
/*     */     }
/*     */ 
/* 659 */     return new SimpleDelivery(this, ref, true, false);
/*     */   }
/*     */ 
/*     */   protected boolean checkAndSchedule(MessageReference ref)
/*     */   {
/* 665 */     if (ref.getScheduledDeliveryTime() > System.currentTimeMillis())
/*     */     {
/* 667 */       if (this.trace) log.trace("Scheduling delivery for " + ref + " to occur at " + ref.getScheduledDeliveryTime());
/*     */ 
/* 671 */       synchronized (this.scheduledDeliveries)
/*     */       {
/* 673 */         Timeout timeout = MessagingTimeoutFactory.instance.getFactory().schedule(ref.getScheduledDeliveryTime(), new DeliverRefTimeoutTarget(ref));
/*     */ 
/* 677 */         this.scheduledDeliveries.add(timeout);
/*     */       }
/*     */ 
/* 680 */       return true;
/*     */     }
/*     */ 
/* 684 */     return false;
/*     */   }
/*     */ 
/*     */   protected void acknowledgeInternal(Delivery d, Transaction tx, boolean persist)
/*     */     throws Exception
/*     */   {
/* 690 */     if (tx == null)
/*     */     {
/* 692 */       if ((persist) && (this.recoverable) && (d.getReference().getMessage().isReliable()))
/*     */       {
/* 694 */         this.pm.removeReference(this.channelID, d.getReference(), null);
/*     */       }
/*     */ 
/* 697 */       if (!d.isRecovered())
/*     */       {
/* 699 */         this.deliveringCount.decrement();
/*     */       }
/*     */     }
/*     */     else
/*     */     {
/* 704 */       getCallback(tx).addDelivery(d);
/*     */ 
/* 706 */       if (this.trace) log.trace(this + " added " + d + " to memory on transaction " + tx);
/*     */ 
/* 708 */       if ((this.recoverable) && (d.getReference().getMessage().isReliable()))
/*     */       {
/* 710 */         this.pm.removeReference(this.channelID, d.getReference(), tx);
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   protected InMemoryCallback getCallback(Transaction tx)
/*     */   {
/* 717 */     InMemoryCallback callback = (InMemoryCallback)tx.getCallback(this);
/*     */ 
/* 719 */     if (callback == null)
/*     */     {
/* 721 */       callback = new InMemoryCallback(null);
/*     */ 
/* 723 */       tx.addCallback(callback, this);
/*     */     }
/*     */ 
/* 726 */     return callback;
/*     */   }
/*     */ 
/*     */   protected MessageReference removeFirstInMemory() throws Exception
/*     */   {
/* 731 */     MessageReference result = (MessageReference)this.messageRefs.removeFirst();
/*     */ 
/* 733 */     return result;
/*     */   }
/*     */ 
/*     */   protected void addReferenceInMemory(MessageReference ref) throws Exception
/*     */   {
/* 738 */     this.messageRefs.addLast(ref, ref.getMessage().getPriority());
/*     */ 
/* 740 */     if (this.trace) log.trace(this + " added " + ref + " non-transactionally in memory");
/*     */   }
/*     */ 
/*     */   protected boolean getReceiversReady()
/*     */   {
/* 745 */     return this.receiversReady;
/*     */   }
/*     */ 
/*     */   protected void setReceiversReady(boolean receiversReady)
/*     */   {
/* 750 */     this.receiversReady = receiversReady;
/*     */   }
/*     */ 
/*     */   private MessageReference nextReference(ListIterator iter)
/*     */     throws Throwable
/*     */   {
/*     */     MessageReference ref;
/*     */     MessageReference ref;
/* 759 */     if (iter == null)
/*     */     {
/* 762 */       ref = (MessageReference)this.messageRefs.peekFirst();
/*     */     }
/*     */     else
/*     */     {
/*     */       MessageReference ref;
/* 770 */       if (iter.hasNext())
/*     */       {
/* 772 */         ref = (MessageReference)iter.next();
/*     */       }
/*     */       else
/*     */       {
/* 776 */         ref = null;
/*     */       }
/*     */     }
/*     */ 
/* 780 */     return ref;
/*     */   }
/*     */ 
/*     */   protected void processMessageBeforeStorage(MessageReference reference)
/*     */   {
/*     */   }
/*     */ 
/*     */   protected void checkClosed()
/*     */   {
/* 919 */     if (this.distributor == null)
/*     */     {
/* 921 */       throw new IllegalStateException(this + " closed");
/*     */     }
/*     */   }
/*     */ 
/*     */   private class DeliverRefTimeoutTarget
/*     */     implements TimeoutTarget
/*     */   {
/*     */     private MessageReference ref;
/*     */ 
/*     */     public DeliverRefTimeoutTarget(MessageReference ref)
/*     */     {
/* 935 */       this.ref = ref;
/*     */     }
/*     */ 
/*     */     public void timedOut(Timeout timeout)
/*     */     {
/* 940 */       if (ChannelSupport.this.trace) ChannelSupport.log.trace("Scheduled delivery timeout " + this.ref);
/*     */ 
/* 942 */       synchronized (ChannelSupport.this.scheduledDeliveries)
/*     */       {
/* 944 */         boolean removed = ChannelSupport.this.scheduledDeliveries.remove(timeout);
/*     */ 
/* 946 */         if (!removed)
/*     */         {
/* 948 */           throw new IllegalStateException("Failed to remove timeout " + timeout);
/*     */         }
/*     */       }
/*     */ 
/* 952 */       this.ref.setScheduledDeliveryTime(0L);
/*     */ 
/* 954 */       boolean delivered = false;
/*     */ 
/* 956 */       if (ChannelSupport.this.distributor.getNumberOfReceivers() > 0)
/*     */       {
/* 958 */         delivered = ChannelSupport.this.deliverScheduled(this.ref);
/*     */       }
/*     */ 
/* 961 */       if (!delivered)
/*     */       {
/*     */         try
/*     */         {
/* 965 */           ChannelSupport.this.cancelInternal(this.ref);
/*     */         }
/*     */         catch (Exception e)
/*     */         {
/* 969 */           ChannelSupport.log.error("Failed to cancel", e);
/*     */         }
/*     */ 
/*     */       }
/* 974 */       else if (ChannelSupport.this.trace) ChannelSupport.log.trace("Delivered scheduled delivery at " + System.currentTimeMillis() + " for " + this.ref);
/*     */     }
/*     */   }
/*     */ 
/*     */   private class InMemoryCallback
/*     */     implements TxCallback
/*     */   {
/*     */     private List refsToAdd;
/*     */     private List deliveriesToRemove;
/*     */ 
/*     */     private InMemoryCallback()
/*     */     {
/* 793 */       this.refsToAdd = new ArrayList();
/*     */ 
/* 795 */       this.deliveriesToRemove = new ArrayList();
/*     */     }
/*     */ 
/*     */     private void addRef(MessageReference ref)
/*     */     {
/* 800 */       this.refsToAdd.add(ref);
/*     */     }
/*     */ 
/*     */     private void addDelivery(Delivery del)
/*     */     {
/* 805 */       this.deliveriesToRemove.add(del);
/*     */     }
/*     */ 
/*     */     public void beforePrepare()
/*     */     {
/*     */     }
/*     */ 
/*     */     public void beforeCommit(boolean onePhase)
/*     */     {
/*     */     }
/*     */ 
/*     */     public void beforeRollback(boolean onePhase)
/*     */     {
/*     */     }
/*     */ 
/*     */     public void afterPrepare()
/*     */     {
/*     */     }
/*     */ 
/*     */     public void afterCommit(boolean onePhase)
/*     */       throws Exception
/*     */     {
/*     */       try
/*     */       {
/* 834 */         boolean promptDelivery = false;
/*     */ 
/* 836 */         for (Iterator i = this.refsToAdd.iterator(); i.hasNext(); )
/*     */         {
/* 838 */           MessageReference ref = (MessageReference)i.next();
/*     */ 
/* 840 */           if (ChannelSupport.this.checkAndSchedule(ref))
/*     */           {
/* 842 */             if (ChannelSupport.this.trace) ChannelSupport.log.trace(this + ": scheduled " + ref);
/*     */           }
/*     */           else
/*     */           {
/* 846 */             if (ChannelSupport.this.trace) ChannelSupport.log.trace(this + ": adding " + ref + " to memory");
/*     */ 
/*     */             try
/*     */             {
/* 850 */               synchronized (ChannelSupport.this.lock)
/*     */               {
/* 852 */                 ChannelSupport.this.addReferenceInMemory(ref);
/*     */               }
/*     */             }
/*     */             catch (Throwable t)
/*     */             {
/* 857 */               throw new TransactionException("Failed to add reference", t);
/*     */             }
/*     */ 
/* 861 */             promptDelivery = true;
/*     */           }
/*     */ 
/*     */         }
/*     */ 
/* 867 */         for (Iterator i = this.deliveriesToRemove.iterator(); i.hasNext(); )
/*     */         {
/* 869 */           Delivery del = (Delivery)i.next();
/*     */ 
/* 871 */           if (ChannelSupport.this.trace) ChannelSupport.log.trace(this + " removing " + del + " after commit");
/*     */ 
/* 873 */           if (!del.isRecovered())
/*     */           {
/* 875 */             ChannelSupport.this.deliveringCount.decrement();
/*     */           }
/*     */ 
/*     */         }
/*     */ 
/* 880 */         if (promptDelivery)
/*     */         {
/* 882 */           synchronized (ChannelSupport.this.lock)
/*     */           {
/* 884 */             ChannelSupport.this.deliverInternal();
/*     */           }
/*     */         }
/*     */       }
/*     */       catch (Throwable t)
/*     */       {
/* 890 */         ChannelSupport.log.error("failed to commit", t);
/* 891 */         throw new Exception("Failed to commit", t);
/*     */       }
/*     */     }
/*     */ 
/*     */     public void afterRollback(boolean onePhase)
/*     */       throws Exception
/*     */     {
/*     */     }
/*     */ 
/*     */     public String toString()
/*     */     {
/* 903 */       return ChannelSupport.this + ".InMemoryCallback[" + Integer.toHexString(hashCode()) + "]";
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.messaging.core.impl.ChannelSupport
 * JD-Core Version:    0.6.0
 */