/*     */ package org.jboss.resource.adapter.jms.inflow;
/*     */ 
/*     */ import javax.jms.Connection;
/*     */ import javax.jms.JMSException;
/*     */ import javax.jms.Message;
/*     */ import javax.jms.MessageListener;
/*     */ import javax.jms.ServerSession;
/*     */ import javax.jms.Session;
/*     */ import javax.jms.XAConnection;
/*     */ import javax.jms.XASession;
/*     */ import javax.resource.spi.endpoint.MessageEndpoint;
/*     */ import javax.resource.spi.endpoint.MessageEndpointFactory;
/*     */ import javax.resource.spi.work.Work;
/*     */ import javax.resource.spi.work.WorkEvent;
/*     */ import javax.resource.spi.work.WorkException;
/*     */ import javax.resource.spi.work.WorkListener;
/*     */ import javax.resource.spi.work.WorkManager;
/*     */ import javax.transaction.Transaction;
/*     */ import javax.transaction.TransactionManager;
/*     */ import javax.transaction.xa.XAResource;
/*     */ import org.jboss.logging.Logger;
/*     */ 
/*     */ public class JmsServerSession
/*     */   implements ServerSession, MessageListener, Work, WorkListener
/*     */ {
/*  57 */   private static final Logger log = Logger.getLogger(JmsServerSession.class);
/*     */   JmsServerSessionPool pool;
/*     */   boolean transacted;
/*     */   int acknowledge;
/*     */   Session session;
/*     */   XASession xaSession;
/*     */   MessageEndpoint endpoint;
/*     */   DLQHandler dlqHandler;
/*     */   TransactionDemarcationStrategy txnStrategy;
/*     */ 
/*     */   public JmsServerSession(JmsServerSessionPool pool)
/*     */   {
/*  90 */     this.pool = pool;
/*     */   }
/*     */ 
/*     */   public void setup()
/*     */     throws Exception
/*     */   {
/*  99 */     JmsActivation activation = this.pool.getActivation();
/* 100 */     JmsActivationSpec spec = activation.getActivationSpec();
/*     */ 
/* 102 */     this.dlqHandler = activation.getDLQHandler();
/*     */ 
/* 104 */     Connection connection = activation.getConnection();
/*     */ 
/* 107 */     if (((connection instanceof XAConnection)) && (activation.isDeliveryTransacted()))
/*     */     {
/* 110 */       this.xaSession = ((XAConnection)connection).createXASession();
/* 111 */       this.session = this.xaSession.getSession();
/*     */     }
/*     */     else {
/* 114 */       this.transacted = spec.isSessionTransacted();
/* 115 */       this.acknowledge = spec.getAcknowledgeModeInt();
/* 116 */       this.session = connection.createSession(this.transacted, this.acknowledge);
/*     */     }
/*     */ 
/* 120 */     MessageEndpointFactory endpointFactory = activation.getMessageEndpointFactory();
/*     */ 
/* 122 */     XAResource xaResource = null;
/*     */ 
/* 124 */     if ((activation.isDeliveryTransacted()) && (this.xaSession != null)) {
/* 125 */       xaResource = this.xaSession.getXAResource();
/*     */     }
/* 127 */     this.endpoint = endpointFactory.createEndpoint(xaResource);
/*     */ 
/* 130 */     this.session.setMessageListener(this);
/*     */   }
/*     */ 
/*     */   public void teardown()
/*     */   {
/*     */     try
/*     */     {
/* 140 */       if (this.endpoint != null)
/* 141 */         this.endpoint.release();
/*     */     }
/*     */     catch (Throwable t) {
/* 144 */       log.debug("Error releasing endpoint " + this.endpoint, t);
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 149 */       if (this.xaSession != null)
/* 150 */         this.xaSession.close();
/*     */     }
/*     */     catch (Throwable t) {
/* 153 */       log.debug("Error releasing xaSession " + this.xaSession, t);
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 158 */       if (this.session != null)
/* 159 */         this.session.close();
/*     */     }
/*     */     catch (Throwable t) {
/* 162 */       log.debug("Error releasing session " + this.session, t);
/*     */     }
/*     */   }
/*     */ 
/*     */   public void onMessage(Message message)
/*     */   {
/*     */     try
/*     */     {
/* 170 */       this.endpoint.beforeDelivery(JmsActivation.ONMESSAGE);
/*     */       try
/*     */       {
/* 174 */         if ((this.dlqHandler == null) || (!this.dlqHandler.handleRedeliveredMessage(message)))
/*     */         {
/* 177 */           MessageListener listener = (MessageListener)this.endpoint;
/* 178 */           listener.onMessage(message);
/*     */         }
/*     */       }
/*     */       finally {
/* 182 */         this.endpoint.afterDelivery();
/*     */ 
/* 184 */         if (this.dlqHandler != null) {
/* 185 */           this.dlqHandler.messageDelivered(message);
/*     */         }
/*     */       }
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 191 */       log.error("Unexpected error delivering message " + message, t);
/*     */ 
/* 193 */       if (this.txnStrategy != null)
/* 194 */         this.txnStrategy.error();
/*     */     }
/*     */   }
/*     */ 
/*     */   public Session getSession()
/*     */     throws JMSException
/*     */   {
/* 202 */     return this.session;
/*     */   }
/*     */ 
/*     */   public void start() throws JMSException
/*     */   {
/* 207 */     JmsActivation activation = this.pool.getActivation();
/* 208 */     WorkManager workManager = activation.getWorkManager();
/*     */     try
/*     */     {
/* 211 */       workManager.scheduleWork(this, 0L, null, this);
/*     */     }
/*     */     catch (WorkException e) {
/* 214 */       log.error("Unable to schedule work", e);
/* 215 */       throw new JMSException("Unable to schedule work: " + e.toString());
/*     */     }
/*     */   }
/*     */ 
/*     */   public void run()
/*     */   {
/*     */     try
/*     */     {
/* 224 */       this.txnStrategy = createTransactionDemarcation();
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 228 */       log.error("Error creating transaction demarcation. Cannot continue.");
/* 229 */       return;
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 234 */       this.session.run();
/*     */     }
/*     */     catch (Throwable t) {
/* 237 */       if (this.txnStrategy != null)
/* 238 */         this.txnStrategy.error();
/*     */     }
/*     */     finally
/*     */     {
/* 242 */       if (this.txnStrategy != null) {
/* 243 */         this.txnStrategy.end();
/*     */       }
/* 245 */       this.txnStrategy = null;
/*     */     }
/*     */   }
/*     */ 
/*     */   private TransactionDemarcationStrategy createTransactionDemarcation()
/*     */   {
/* 252 */     return new DemarcationStrategyFactory(null).getStrategy();
/*     */   }
/*     */ 
/*     */   public void release()
/*     */   {
/*     */   }
/*     */ 
/*     */   public void workAccepted(WorkEvent e)
/*     */   {
/*     */   }
/*     */ 
/*     */   public void workCompleted(WorkEvent e)
/*     */   {
/* 266 */     this.pool.returnServerSession(this);
/*     */   }
/*     */ 
/*     */   public void workRejected(WorkEvent e)
/*     */   {
/* 271 */     this.pool.returnServerSession(this);
/*     */   }
/*     */ 
/*     */   public void workStarted(WorkEvent e)
/*     */   {
/*     */   }
/*     */ 
/*     */   private class XATransactionDemarcationStrategy
/*     */     implements JmsServerSession.TransactionDemarcationStrategy
/*     */   {
/* 382 */     boolean trace = JmsServerSession.log.isTraceEnabled();
/*     */ 
/* 384 */     Transaction trans = null;
/*     */ 
/* 386 */     TransactionManager tm = JmsServerSession.this.pool.getActivation().getTransactionManager();
/*     */ 
/*     */     public XATransactionDemarcationStrategy() throws Throwable
/*     */     {
/* 390 */       int timeout = JmsServerSession.this.pool.getActivation().getActivationSpec().getTransactionTimeout();
/*     */ 
/* 393 */       if (timeout > 0)
/*     */       {
/* 395 */         JmsServerSession.log.trace("Setting transactionTimeout for JMSSessionPool to " + timeout);
/*     */ 
/* 397 */         this.tm.setTransactionTimeout(timeout);
/*     */       }
/*     */ 
/* 401 */       this.tm.begin();
/*     */       try
/*     */       {
/* 405 */         this.trans = this.tm.getTransaction();
/*     */ 
/* 407 */         if (this.trace) {
/* 408 */           JmsServerSession.log.trace(JmsServerSession.this + " using tx=" + this.trans);
/*     */         }
/* 410 */         if (JmsServerSession.this.xaSession != null)
/*     */         {
/* 412 */           XAResource res = JmsServerSession.this.xaSession.getXAResource();
/*     */ 
/* 414 */           if (!this.trans.enlistResource(res))
/*     */           {
/* 416 */             throw new JMSException("could not enlist resource");
/*     */           }
/* 418 */           if (this.trace)
/* 419 */             JmsServerSession.log.trace(JmsServerSession.this + " XAResource '" + res + "' enlisted.");
/*     */         }
/*     */       }
/*     */       catch (Throwable t)
/*     */       {
/*     */         try
/*     */         {
/* 426 */           this.tm.rollback();
/*     */         }
/*     */         catch (Throwable ignored) {
/* 429 */           JmsServerSession.log.trace(JmsServerSession.this + " ignored error rolling back after failed enlist", ignored);
/*     */         }
/*     */ 
/* 433 */         throw t;
/*     */       }
/*     */     }
/*     */ 
/*     */     public void error()
/*     */     {
/*     */       try
/*     */       {
/* 444 */         if (this.trace) {
/* 445 */           JmsServerSession.log.trace(JmsServerSession.this + " using TM to mark TX for rollback tx=" + this.trans);
/*     */         }
/* 447 */         this.trans.setRollbackOnly();
/*     */       }
/*     */       catch (Throwable t) {
/* 450 */         JmsServerSession.log.error(JmsServerSession.this + " failed to set rollback only", t);
/*     */       }
/*     */     }
/*     */ 
/*     */     public void end()
/*     */     {
/*     */       try
/*     */       {
/* 464 */         Transaction currentTx = this.tm.getTransaction();
/* 465 */         if (!this.trans.equals(currentTx)) {
/* 466 */           throw new IllegalStateException("Wrong tx association: expected " + this.trans + " was " + currentTx);
/*     */         }
/*     */ 
/* 471 */         if (this.trans.getStatus() == 1)
/*     */         {
/* 473 */           if (this.trace) {
/* 474 */             JmsServerSession.log.trace(JmsServerSession.this + " rolling back JMS transaction tx=" + this.trans);
/*     */           }
/*     */ 
/* 477 */           this.tm.rollback();
/*     */ 
/* 482 */           if ((JmsServerSession.this.xaSession == null) && (JmsServerSession.this.pool.getActivation().isDeliveryTransacted()))
/*     */           {
/* 485 */             JmsServerSession.this.session.rollback();
/*     */           }
/*     */ 
/*     */         }
/* 489 */         else if (this.trans.getStatus() == 0)
/*     */         {
/* 495 */           if (this.trace) {
/* 496 */             JmsServerSession.log.trace(JmsServerSession.this + " commiting the JMS transaction tx=" + this.trans);
/*     */           }
/* 498 */           this.tm.commit();
/*     */ 
/* 502 */           if ((JmsServerSession.this.xaSession == null) && (JmsServerSession.this.pool.getActivation().isDeliveryTransacted()))
/*     */           {
/* 505 */             JmsServerSession.this.session.commit();
/*     */           }
/*     */         }
/*     */         else
/*     */         {
/* 510 */           this.tm.suspend();
/*     */ 
/* 512 */           if ((JmsServerSession.this.xaSession == null) && (JmsServerSession.this.pool.getActivation().isDeliveryTransacted()))
/*     */           {
/* 515 */             JmsServerSession.this.session.rollback();
/*     */           }
/*     */         }
/*     */ 
/*     */       }
/*     */       catch (Throwable t)
/*     */       {
/* 522 */         JmsServerSession.log.error(JmsServerSession.this + " failed to commit/rollback", t);
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   private class LocalDemarcationStrategy
/*     */     implements JmsServerSession.TransactionDemarcationStrategy
/*     */   {
/*     */     private LocalDemarcationStrategy()
/*     */     {
/*     */     }
/*     */ 
/*     */     public void end()
/*     */     {
/* 323 */       JmsActivationSpec spec = JmsServerSession.this.pool.getActivation().getActivationSpec();
/*     */ 
/* 326 */       if (spec.isSessionTransacted())
/*     */       {
/* 328 */         if (JmsServerSession.this.session != null)
/*     */         {
/*     */           try
/*     */           {
/* 332 */             JmsServerSession.this.session.commit();
/*     */           }
/*     */           catch (JMSException e) {
/* 335 */             JmsServerSession.log.error("Failed to commit session transaction", e);
/*     */           }
/*     */         }
/*     */       }
/*     */     }
/*     */ 
/*     */     public void error()
/*     */     {
/* 343 */       JmsActivationSpec spec = JmsServerSession.this.pool.getActivation().getActivationSpec();
/*     */ 
/* 346 */       if (spec.isSessionTransacted())
/*     */       {
/* 348 */         if (JmsServerSession.this.session != null)
/*     */         {
/*     */           try
/*     */           {
/* 362 */             if ((JmsServerSession.this.pool.getActivation().isDeliveryTransacted()) || (spec.getRedeliverUnspecified()))
/*     */             {
/* 365 */               JmsServerSession.this.session.rollback();
/*     */             }
/*     */           }
/*     */           catch (JMSException e)
/*     */           {
/* 370 */             JmsServerSession.log.error("Failed to rollback session transaction", e);
/*     */           }
/*     */         }
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   private static abstract interface TransactionDemarcationStrategy
/*     */   {
/*     */     public abstract void error();
/*     */ 
/*     */     public abstract void end();
/*     */   }
/*     */ 
/*     */   private class DemarcationStrategyFactory
/*     */   {
/*     */     private DemarcationStrategyFactory()
/*     */     {
/*     */     }
/*     */ 
/*     */     JmsServerSession.TransactionDemarcationStrategy getStrategy()
/*     */     {
/* 283 */       JmsServerSession.TransactionDemarcationStrategy current = null;
/* 284 */       JmsActivationSpec spec = JmsServerSession.this.pool.getActivation().getActivationSpec();
/*     */ 
/* 286 */       JmsActivation activation = JmsServerSession.this.pool.getActivation();
/*     */ 
/* 288 */       if ((activation.isDeliveryTransacted()) && (JmsServerSession.this.xaSession != null))
/*     */       {
/*     */         try
/*     */         {
/* 292 */           current = new JmsServerSession.XATransactionDemarcationStrategy(JmsServerSession.this);
/*     */         }
/*     */         catch (Throwable t) {
/* 295 */           JmsServerSession.log.error(this + " error creating transaction demarcation ", t);
/*     */         }
/*     */ 
/*     */       }
/*     */       else
/*     */       {
/* 301 */         return new JmsServerSession.LocalDemarcationStrategy(JmsServerSession.this, null);
/*     */       }
/*     */ 
/* 305 */       return current;
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.resource.adapter.jms.inflow.JmsServerSession
 * JD-Core Version:    0.6.0
 */