/*     */ package org.jboss.resource.adapter.jms.inflow.dlq;
/*     */ 
/*     */ import java.util.Enumeration;
/*     */ import java.util.HashMap;
/*     */ import java.util.Iterator;
/*     */ import java.util.Set;
/*     */ import javax.jms.Destination;
/*     */ import javax.jms.ExceptionListener;
/*     */ import javax.jms.JMSException;
/*     */ import javax.jms.Message;
/*     */ import javax.jms.Queue;
/*     */ import javax.jms.QueueConnection;
/*     */ import javax.jms.QueueConnectionFactory;
/*     */ import javax.jms.QueueSender;
/*     */ import javax.jms.QueueSession;
/*     */ import javax.naming.Context;
/*     */ import org.jboss.jms.jndi.JMSProviderAdapter;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.resource.adapter.jms.inflow.DLQHandler;
/*     */ import org.jboss.resource.adapter.jms.inflow.JmsActivation;
/*     */ import org.jboss.resource.adapter.jms.inflow.JmsActivationSpec;
/*     */ import org.jboss.util.naming.Util;
/*     */ 
/*     */ public abstract class AbstractDLQHandler
/*     */   implements DLQHandler, ExceptionListener
/*     */ {
/*  56 */   protected static final Logger log = Logger.getLogger(AbstractDLQHandler.class);
/*     */   protected JmsActivation activation;
/*     */   protected Queue dlq;
/*     */   protected QueueConnection connection;
/*     */ 
/*     */   public boolean handleRedeliveredMessage(Message msg)
/*     */   {
/*  69 */     boolean handled = handleDelivery(msg);
/*  70 */     if (handled)
/*  71 */       sendToDLQ(msg);
/*  72 */     return handled;
/*     */   }
/*     */ 
/*     */   public void messageDelivered(Message msg)
/*     */   {
/*     */   }
/*     */ 
/*     */   public void setup(JmsActivation activation, Context ctx) throws Exception
/*     */   {
/*  81 */     this.activation = activation;
/*  82 */     setupDLQDestination(ctx);
/*  83 */     setupDLQConnection(ctx);
/*     */   }
/*     */ 
/*     */   public void teardown()
/*     */   {
/*  88 */     teardownDLQConnection();
/*  89 */     teardownDLQDestination();
/*     */   }
/*     */ 
/*     */   public void onException(JMSException exception)
/*     */   {
/*  94 */     this.activation.handleFailure(exception);
/*     */   }
/*     */ 
/*     */   protected void setupDLQDestination(Context ctx)
/*     */     throws Exception
/*     */   {
/* 105 */     String name = this.activation.getActivationSpec().getDLQJNDIName();
/* 106 */     this.dlq = ((Queue)Util.lookup(ctx, name, Queue.class));
/*     */   }
/*     */ 
/*     */   protected void teardownDLQDestination()
/*     */   {
/*     */   }
/*     */ 
/*     */   protected void setupDLQConnection(Context ctx)
/*     */     throws Exception
/*     */   {
/* 124 */     JmsActivationSpec spec = this.activation.getActivationSpec();
/* 125 */     String user = spec.getDLQUser();
/* 126 */     String pass = spec.getDLQPassword();
/* 127 */     String clientID = spec.getDLQClientID();
/* 128 */     JMSProviderAdapter adapter = this.activation.getProviderAdapter();
/* 129 */     String queueFactoryRef = adapter.getQueueFactoryRef();
/* 130 */     log.debug("Attempting to lookup dlq connection factory " + queueFactoryRef);
/* 131 */     QueueConnectionFactory qcf = (QueueConnectionFactory)Util.lookup(ctx, queueFactoryRef, QueueConnectionFactory.class);
/* 132 */     log.debug("Got dlq connection factory " + qcf + " from " + queueFactoryRef);
/* 133 */     log.debug("Attempting to create queue connection with user " + user);
/* 134 */     if (user != null)
/* 135 */       this.connection = qcf.createQueueConnection(user, pass);
/*     */     else
/* 137 */       this.connection = qcf.createQueueConnection();
/*     */     try
/*     */     {
/* 140 */       if (clientID != null)
/* 141 */         this.connection.setClientID(clientID);
/* 142 */       this.connection.setExceptionListener(this);
/* 143 */       log.debug("Using queue connection " + this.connection);
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/*     */       try
/*     */       {
/* 149 */         this.connection.close();
/*     */       }
/*     */       catch (Exception e)
/*     */       {
/* 153 */         log.trace("Ignored error closing connection", e);
/*     */       }
/* 155 */       this.connection = null;
/* 156 */       if ((t instanceof Exception))
/* 157 */         throw ((Exception)t);
/* 158 */       throw new RuntimeException("Error configuring queue connection", t);
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void teardownDLQConnection()
/*     */   {
/*     */     try
/*     */     {
/* 169 */       if (this.connection != null)
/*     */       {
/* 171 */         log.debug("Closing the " + this.connection);
/* 172 */         this.connection.close();
/*     */       }
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 177 */       log.debug("Error closing the connection " + this.connection, t);
/*     */     }
/* 179 */     this.connection = null;
/*     */   }
/*     */ 
/*     */   protected abstract boolean handleDelivery(Message paramMessage);
/*     */ 
/*     */   protected void warnDLQ(Message msg, int count, int max)
/*     */   {
/* 199 */     log.warn("Message redelivered=" + count + " max=" + max + " sending it to the dlq " + msg);
/*     */   }
/*     */ 
/*     */   protected void sendToDLQ(Message msg)
/*     */   {
/* 209 */     int deliveryMode = getDeliveryMode(msg);
/* 210 */     int priority = getPriority(msg);
/* 211 */     long timeToLive = getTimeToLive(msg);
/*     */ 
/* 214 */     if (timeToLive < 0L)
/*     */     {
/* 216 */       if (log.isTraceEnabled())
/* 217 */         log.trace("Not sending the message to the DLQ, it has expired " + msg);
/* 218 */       return;
/*     */     }
/*     */ 
/* 221 */     Message copy = makeWritable(msg);
/* 222 */     if (copy != null)
/* 223 */       doSend(copy, deliveryMode, priority, timeToLive);
/*     */   }
/*     */ 
/*     */   protected int getDeliveryMode(Message msg)
/*     */   {
/*     */     try
/*     */     {
/* 236 */       return msg.getJMSDeliveryMode();
/*     */     }
/*     */     catch (Throwable t) {
/*     */     }
/* 240 */     return 2;
/*     */   }
/*     */ 
/*     */   protected int getPriority(Message msg)
/*     */   {
/*     */     try
/*     */     {
/* 254 */       return msg.getJMSPriority();
/*     */     }
/*     */     catch (Throwable t) {
/*     */     }
/* 258 */     return 4;
/*     */   }
/*     */ 
/*     */   protected long getTimeToLive(Message msg)
/*     */   {
/*     */     try
/*     */     {
/* 272 */       long expires = msg.getJMSExpiration();
/* 273 */       if (expires == 0L)
/* 274 */         return 0L;
/* 275 */       return expires - System.currentTimeMillis();
/*     */     }
/*     */     catch (Throwable t) {
/*     */     }
/* 279 */     return 0L;
/*     */   }
/*     */ 
/*     */   protected Message makeWritable(Message msg)
/*     */   {
/* 291 */     boolean trace = log.isTraceEnabled();
/*     */     try
/*     */     {
/* 295 */       HashMap tmp = new HashMap();
/*     */ 
/* 298 */       for (Enumeration en = msg.getPropertyNames(); en.hasMoreElements(); )
/*     */       {
/* 300 */         String key = (String)en.nextElement();
/* 301 */         tmp.put(key, msg.getObjectProperty(key));
/*     */       }
/*     */ 
/* 305 */       msg.clearProperties();
/*     */ 
/* 307 */       for (Iterator i = tmp.keySet().iterator(); i.hasNext(); )
/*     */       {
/* 309 */         String key = (String)i.next();
/*     */         try
/*     */         {
/* 312 */           msg.setObjectProperty(key, tmp.get(key));
/*     */         }
/*     */         catch (JMSException ignored)
/*     */         {
/* 316 */           if (trace) {
/* 317 */             log.trace("Could not copy message property " + key, ignored);
/*     */           }
/*     */         }
/*     */       }
/* 321 */       msg.setStringProperty("JBOSS_ORIG_MESSAGEID", msg.getJMSMessageID());
/* 322 */       Destination destination = msg.getJMSDestination();
/* 323 */       if (destination != null) {
/* 324 */         msg.setStringProperty("JBOSS_ORIG_DESTINATION", destination.toString());
/*     */       }
/* 326 */       return msg;
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 330 */       log.error("Unable to make writable " + msg, t);
/* 331 */     }return null;
/*     */   }
/*     */ 
/*     */   protected void doSend(Message msg, int deliveryMode, int priority, long timeToLive)
/*     */   {
/* 342 */     QueueSession session = null;
/*     */     try
/*     */     {
/* 345 */       session = this.connection.createQueueSession(false, 1);
/* 346 */       QueueSender sender = session.createSender(this.dlq);
/* 347 */       sender.send(msg, deliveryMode, priority, timeToLive);
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 351 */       handleSendError(msg, t);
/*     */     }
/*     */     finally
/*     */     {
/* 355 */       if (session != null)
/*     */       {
/*     */         try
/*     */         {
/* 359 */           session.close();
/*     */         }
/*     */         catch (Throwable t)
/*     */         {
/* 363 */           log.trace("Ignored ", t);
/*     */         }
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void handleSendError(Message msg, Throwable t)
/*     */   {
/* 377 */     log.error("DLQ " + this.dlq + " error sending message " + msg, t);
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.resource.adapter.jms.inflow.dlq.AbstractDLQHandler
 * JD-Core Version:    0.6.0
 */