/*     */ package org.jboss.util.collection;
/*     */ 
/*     */ import java.lang.ref.ReferenceQueue;
/*     */ import java.lang.ref.WeakReference;
/*     */ import java.util.AbstractCollection;
/*     */ import java.util.AbstractSet;
/*     */ import java.util.ArrayList;
/*     */ import java.util.Collection;
/*     */ import java.util.ConcurrentModificationException;
/*     */ import java.util.Iterator;
/*     */ import java.util.Map;
/*     */ import java.util.Map.Entry;
/*     */ import java.util.NoSuchElementException;
/*     */ import java.util.Set;
/*     */ 
/*     */ public class WeakIdentityHashMap
/*     */   implements Map
/*     */ {
/*     */   private static final int DEFAULT_INITIAL_CAPACITY = 16;
/*     */   private static final int MAXIMUM_CAPACITY = 1073741824;
/*     */   private static final float DEFAULT_LOAD_FACTOR = 0.75F;
/*     */   private Entry[] table;
/*     */   private int size;
/*     */   private int threshold;
/*     */   private final float loadFactor;
/*  91 */   private final ReferenceQueue queue = new ReferenceQueue();
/*     */   private volatile int modCount;
/* 107 */   volatile transient Set keySet = null;
/* 108 */   volatile transient Collection values = null;
/*     */ 
/* 181 */   private static final Object NULL_KEY = new Object();
/*     */ 
/* 725 */   private transient Set entrySet = null;
/*     */ 
/*     */   public WeakIdentityHashMap(int initialCapacity, float loadFactor)
/*     */   {
/* 122 */     if (initialCapacity < 0) {
/* 123 */       throw new IllegalArgumentException("Illegal Initial Capacity: " + initialCapacity);
/*     */     }
/* 125 */     if (initialCapacity > 1073741824) {
/* 126 */       initialCapacity = 1073741824;
/*     */     }
/* 128 */     if ((loadFactor <= 0.0F) || (Float.isNaN(loadFactor))) {
/* 129 */       throw new IllegalArgumentException("Illegal Load factor: " + loadFactor);
/*     */     }
/* 131 */     int capacity = 1;
/* 132 */     while (capacity < initialCapacity)
/* 133 */       capacity <<= 1;
/* 134 */     this.table = new Entry[capacity];
/* 135 */     this.loadFactor = loadFactor;
/* 136 */     this.threshold = (int)(capacity * loadFactor);
/*     */   }
/*     */ 
/*     */   public WeakIdentityHashMap(int initialCapacity)
/*     */   {
/* 148 */     this(initialCapacity, 0.75F);
/*     */   }
/*     */ 
/*     */   public WeakIdentityHashMap()
/*     */   {
/* 156 */     this.loadFactor = 0.75F;
/* 157 */     this.threshold = 16;
/* 158 */     this.table = new Entry[16];
/*     */   }
/*     */ 
/*     */   public WeakIdentityHashMap(Map t)
/*     */   {
/* 171 */     this(Math.max((int)(t.size() / 0.75F) + 1, 16), 0.75F);
/*     */ 
/* 173 */     putAll(t);
/*     */   }
/*     */ 
/*     */   private static Object maskNull(Object key)
/*     */   {
/* 187 */     return key == null ? NULL_KEY : key;
/*     */   }
/*     */ 
/*     */   private static Object unmaskNull(Object key)
/*     */   {
/* 194 */     return key == NULL_KEY ? null : key;
/*     */   }
/*     */ 
/*     */   int hash(Object x)
/*     */   {
/* 201 */     int h = System.identityHashCode(x);
/* 202 */     return h - (h << 7);
/*     */   }
/*     */ 
/*     */   static int indexFor(int h, int length)
/*     */   {
/* 209 */     return h & length - 1;
/*     */   }
/*     */ 
/*     */   private void expungeStaleEntries()
/*     */   {
/*     */     Object r;
/* 217 */     while ((r = this.queue.poll()) != null) {
/* 218 */       Entry e = (Entry)r;
/* 219 */       int h = e.hash;
/* 220 */       int i = indexFor(h, this.table.length);
/*     */ 
/* 222 */       Entry prev = this.table[i];
/* 223 */       Entry p = prev;
/* 224 */       while (p != null) {
/* 225 */         Entry next = p.next;
/* 226 */         if (p == e) {
/* 227 */           if (prev == e)
/* 228 */             this.table[i] = next;
/*     */           else
/* 230 */             Entry.access$102(prev, next);
/* 231 */           Entry.access$102(e, null);
/* 232 */           Entry.access$202(e, null);
/* 233 */           this.size -= 1;
/* 234 */           break;
/*     */         }
/* 236 */         prev = p;
/* 237 */         p = next;
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   private Entry[] getTable()
/*     */   {
/* 246 */     expungeStaleEntries();
/* 247 */     return this.table;
/*     */   }
/*     */ 
/*     */   public int size()
/*     */   {
/* 257 */     if (this.size == 0)
/* 258 */       return 0;
/* 259 */     expungeStaleEntries();
/* 260 */     return this.size;
/*     */   }
/*     */ 
/*     */   public boolean isEmpty()
/*     */   {
/* 270 */     return size() == 0;
/*     */   }
/*     */ 
/*     */   public Object get(Object key)
/*     */   {
/* 288 */     Object k = maskNull(key);
/* 289 */     int h = hash(k);
/* 290 */     Entry[] tab = getTable();
/* 291 */     int index = indexFor(h, tab.length);
/* 292 */     Entry e = tab[index];
/* 293 */     while (e != null) {
/* 294 */       if ((e.hash == h) && (k == e.get()))
/* 295 */         return e.value;
/* 296 */       e = e.next;
/*     */     }
/* 298 */     return null;
/*     */   }
/*     */ 
/*     */   public boolean containsKey(Object key)
/*     */   {
/* 310 */     return getEntry(key) != null;
/*     */   }
/*     */ 
/*     */   Entry getEntry(Object key)
/*     */   {
/* 318 */     Object k = maskNull(key);
/* 319 */     int h = hash(k);
/* 320 */     Entry[] tab = getTable();
/* 321 */     int index = indexFor(h, tab.length);
/* 322 */     Entry e = tab[index];
/* 323 */     while ((e != null) && ((e.hash != h) || (k != e.get())))
/* 324 */       e = e.next;
/* 325 */     return e;
/*     */   }
/*     */ 
/*     */   public Object put(Object key, Object value)
/*     */   {
/* 341 */     Object k = maskNull(key);
/* 342 */     int h = hash(k);
/* 343 */     Entry[] tab = getTable();
/* 344 */     int i = indexFor(h, tab.length);
/*     */ 
/* 346 */     for (Entry e = tab[i]; e != null; e = e.next) {
/* 347 */       if ((h == e.hash) && (k == e.get())) {
/* 348 */         Object oldValue = e.value;
/* 349 */         if (value != oldValue)
/* 350 */           Entry.access$202(e, value);
/* 351 */         return oldValue;
/*     */       }
/*     */     }
/*     */ 
/* 355 */     this.modCount += 1;
/* 356 */     tab[i] = new Entry(k, value, this.queue, h, tab[i]);
/* 357 */     if (++this.size >= this.threshold)
/* 358 */       resize(tab.length * 2);
/* 359 */     return null;
/*     */   }
/*     */ 
/*     */   void resize(int newCapacity)
/*     */   {
/* 375 */     Entry[] oldTable = getTable();
/* 376 */     int oldCapacity = oldTable.length;
/*     */ 
/* 379 */     if ((this.size < this.threshold) || (oldCapacity > newCapacity)) {
/* 380 */       return;
/*     */     }
/* 382 */     Entry[] newTable = new Entry[newCapacity];
/*     */ 
/* 384 */     transfer(oldTable, newTable);
/* 385 */     this.table = newTable;
/*     */ 
/* 392 */     if (this.size >= this.threshold / 2) {
/* 393 */       this.threshold = (int)(newCapacity * this.loadFactor);
/*     */     } else {
/* 395 */       expungeStaleEntries();
/* 396 */       transfer(newTable, oldTable);
/* 397 */       this.table = oldTable;
/*     */     }
/*     */   }
/*     */ 
/*     */   private void transfer(Entry[] src, Entry[] dest)
/*     */   {
/* 403 */     for (int j = 0; j < src.length; j++) {
/* 404 */       Entry e = src[j];
/* 405 */       src[j] = null;
/* 406 */       while (e != null) {
/* 407 */         Entry next = e.next;
/* 408 */         Object key = e.get();
/* 409 */         if (key == null) {
/* 410 */           Entry.access$102(e, null);
/* 411 */           Entry.access$202(e, null);
/* 412 */           this.size -= 1;
/*     */         } else {
/* 414 */           int i = indexFor(e.hash, dest.length);
/* 415 */           Entry.access$102(e, dest[i]);
/* 416 */           dest[i] = e;
/*     */         }
/* 418 */         e = next;
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   public void putAll(Map t)
/*     */   {
/* 433 */     int n = t.size();
/* 434 */     if (n == 0)
/* 435 */       return;
/* 436 */     if (n >= this.threshold) {
/* 437 */       n = (int)(n / this.loadFactor + 1.0F);
/* 438 */       if (n > 1073741824)
/* 439 */         n = 1073741824;
/* 440 */       int capacity = this.table.length;
/* 441 */       while (capacity < n)
/* 442 */         capacity <<= 1;
/* 443 */       resize(capacity);
/*     */     }
/*     */ 
/* 446 */     for (Iterator i = t.entrySet().iterator(); i.hasNext(); ) {
/* 447 */       Map.Entry e = (Map.Entry)i.next();
/* 448 */       put(e.getKey(), e.getValue());
/*     */     }
/*     */   }
/*     */ 
/*     */   public Object remove(Object key)
/*     */   {
/* 462 */     Object k = maskNull(key);
/* 463 */     int h = hash(k);
/* 464 */     Entry[] tab = getTable();
/* 465 */     int i = indexFor(h, tab.length);
/* 466 */     Entry prev = tab[i];
/* 467 */     Entry e = prev;
/*     */ 
/* 469 */     while (e != null) {
/* 470 */       Entry next = e.next;
/* 471 */       if ((h == e.hash) && (k == e.get())) {
/* 472 */         this.modCount += 1;
/* 473 */         this.size -= 1;
/* 474 */         if (prev == e)
/* 475 */           tab[i] = next;
/*     */         else
/* 477 */           Entry.access$102(prev, next);
/* 478 */         return e.value;
/*     */       }
/* 480 */       prev = e;
/* 481 */       e = next;
/*     */     }
/*     */ 
/* 484 */     return null;
/*     */   }
/*     */ 
/*     */   Entry removeMapping(Object o)
/*     */   {
/* 491 */     if (!(o instanceof Map.Entry))
/* 492 */       return null;
/* 493 */     Entry[] tab = getTable();
/* 494 */     Map.Entry entry = (Map.Entry)o;
/* 495 */     Object k = maskNull(entry.getKey());
/* 496 */     int h = hash(k);
/* 497 */     int i = indexFor(h, tab.length);
/* 498 */     Entry prev = tab[i];
/* 499 */     Entry e = prev;
/*     */ 
/* 501 */     while (e != null) {
/* 502 */       Entry next = e.next;
/* 503 */       if ((h == e.hash) && (e.equals(entry))) {
/* 504 */         this.modCount += 1;
/* 505 */         this.size -= 1;
/* 506 */         if (prev == e)
/* 507 */           tab[i] = next;
/*     */         else
/* 509 */           Entry.access$102(prev, next);
/* 510 */         return e;
/*     */       }
/* 512 */       prev = e;
/* 513 */       e = next;
/*     */     }
/*     */ 
/* 516 */     return null;
/*     */   }
/*     */ 
/*     */   public void clear()
/*     */   {
/* 525 */     while (this.queue.poll() != null);
/* 528 */     this.modCount += 1;
/* 529 */     Entry[] tab = this.table;
/* 530 */     for (int i = 0; i < tab.length; i++)
/* 531 */       tab[i] = null;
/* 532 */     this.size = 0;
/*     */ 
/* 537 */     while (this.queue.poll() != null);
/*     */   }
/*     */ 
/*     */   public boolean containsValue(Object value)
/*     */   {
/* 550 */     if (value == null) {
/* 551 */       return containsNullValue();
/*     */     }
/* 553 */     Entry[] tab = getTable();
/* 554 */     for (int i = tab.length; i-- > 0; )
/* 555 */       for (Entry e = tab[i]; e != null; e = e.next)
/* 556 */         if (value.equals(e.value))
/* 557 */           return true;
/* 558 */     return false;
/*     */   }
/*     */ 
/*     */   private boolean containsNullValue()
/*     */   {
/* 565 */     Entry[] tab = getTable();
/* 566 */     for (int i = tab.length; i-- > 0; )
/* 567 */       for (Entry e = tab[i]; e != null; e = e.next)
/* 568 */         if (e.value == null)
/* 569 */           return true;
/* 570 */     return false;
/*     */   }
/*     */ 
/*     */   public Set keySet()
/*     */   {
/* 739 */     Set ks = this.keySet;
/* 740 */     return this.keySet = new KeySet(null);
/*     */   }
/*     */ 
/*     */   public Collection values()
/*     */   {
/* 796 */     Collection vs = this.values;
/* 797 */     return this.values = new Values(null);
/*     */   }
/*     */ 
/*     */   public Set entrySet()
/*     */   {
/* 846 */     Set es = this.entrySet;
/* 847 */     return this.entrySet = new EntrySet(null);
/*     */   }
/*     */ 
/*     */   static class SimpleEntry
/*     */     implements Map.Entry
/*     */   {
/*     */     Object key;
/*     */     Object value;
/*     */ 
/*     */     public SimpleEntry(Object key, Object value)
/*     */     {
/* 896 */       this.key = key;
/* 897 */       this.value = value;
/*     */     }
/*     */ 
/*     */     public SimpleEntry(Map.Entry e) {
/* 901 */       this.key = e.getKey();
/* 902 */       this.value = e.getValue();
/*     */     }
/*     */ 
/*     */     public Object getKey() {
/* 906 */       return this.key;
/*     */     }
/*     */ 
/*     */     public Object getValue() {
/* 910 */       return this.value;
/*     */     }
/*     */ 
/*     */     public Object setValue(Object value) {
/* 914 */       Object oldValue = this.value;
/* 915 */       this.value = value;
/* 916 */       return oldValue;
/*     */     }
/*     */ 
/*     */     public boolean equals(Object o) {
/* 920 */       if (!(o instanceof Map.Entry))
/* 921 */         return false;
/* 922 */       Map.Entry e = (Map.Entry)o;
/* 923 */       return (eq(this.key, e.getKey())) && (eq(this.value, e.getValue()));
/*     */     }
/*     */ 
/*     */     public int hashCode()
/*     */     {
/* 928 */       return (this.key == null ? 0 : this.key.hashCode()) ^ (this.value == null ? 0 : this.value.hashCode());
/*     */     }
/*     */ 
/*     */     public String toString()
/*     */     {
/* 933 */       return this.key + "=" + this.value;
/*     */     }
/*     */ 
/*     */     private static boolean eq(Object o1, Object o2) {
/* 937 */       return o1 == null ? false : o2 == null ? true : o1.equals(o2);
/*     */     }
/*     */   }
/*     */ 
/*     */   private class EntrySet extends AbstractSet
/*     */   {
/*     */     private EntrySet()
/*     */     {
/*     */     }
/*     */ 
/*     */     public Iterator iterator()
/*     */     {
/* 852 */       return new WeakIdentityHashMap.EntryIterator(WeakIdentityHashMap.this, null);
/*     */     }
/*     */ 
/*     */     public boolean contains(Object o) {
/* 856 */       if (!(o instanceof Map.Entry))
/* 857 */         return false;
/* 858 */       Map.Entry e = (Map.Entry)o;
/* 859 */       Object k = e.getKey();
/* 860 */       WeakIdentityHashMap.Entry candidate = WeakIdentityHashMap.this.getEntry(e.getKey());
/* 861 */       return (candidate != null) && (candidate.equals(e));
/*     */     }
/*     */ 
/*     */     public boolean remove(Object o) {
/* 865 */       return WeakIdentityHashMap.this.removeMapping(o) != null;
/*     */     }
/*     */ 
/*     */     public int size() {
/* 869 */       return WeakIdentityHashMap.this.size();
/*     */     }
/*     */ 
/*     */     public void clear() {
/* 873 */       WeakIdentityHashMap.this.clear();
/*     */     }
/*     */ 
/*     */     public Object[] toArray() {
/* 877 */       Collection c = new ArrayList(size());
/* 878 */       for (Iterator i = iterator(); i.hasNext(); )
/* 879 */         c.add(new WeakIdentityHashMap.SimpleEntry((Map.Entry)i.next()));
/* 880 */       return c.toArray();
/*     */     }
/*     */ 
/*     */     public Object[] toArray(Object[] a) {
/* 884 */       Collection c = new ArrayList(size());
/* 885 */       for (Iterator i = iterator(); i.hasNext(); )
/* 886 */         c.add(new WeakIdentityHashMap.SimpleEntry((Map.Entry)i.next()));
/* 887 */       return c.toArray(a);
/*     */     }
/*     */   }
/*     */ 
/*     */   private class Values extends AbstractCollection
/*     */   {
/*     */     private Values()
/*     */     {
/*     */     }
/*     */ 
/*     */     public Iterator iterator()
/*     */     {
/* 802 */       return new WeakIdentityHashMap.ValueIterator(WeakIdentityHashMap.this, null);
/*     */     }
/*     */ 
/*     */     public int size() {
/* 806 */       return WeakIdentityHashMap.this.size();
/*     */     }
/*     */ 
/*     */     public boolean contains(Object o) {
/* 810 */       return WeakIdentityHashMap.this.containsValue(o);
/*     */     }
/*     */ 
/*     */     public void clear() {
/* 814 */       WeakIdentityHashMap.this.clear();
/*     */     }
/*     */ 
/*     */     public Object[] toArray() {
/* 818 */       Collection c = new ArrayList(size());
/* 819 */       for (Iterator i = iterator(); i.hasNext(); )
/* 820 */         c.add(i.next());
/* 821 */       return c.toArray();
/*     */     }
/*     */ 
/*     */     public Object[] toArray(Object[] a) {
/* 825 */       Collection c = new ArrayList(size());
/* 826 */       for (Iterator i = iterator(); i.hasNext(); )
/* 827 */         c.add(i.next());
/* 828 */       return c.toArray(a);
/*     */     }
/*     */   }
/*     */ 
/*     */   private class KeySet extends AbstractSet
/*     */   {
/*     */     private KeySet()
/*     */     {
/*     */     }
/*     */ 
/*     */     public Iterator iterator()
/*     */     {
/* 745 */       return new WeakIdentityHashMap.KeyIterator(WeakIdentityHashMap.this, null);
/*     */     }
/*     */ 
/*     */     public int size() {
/* 749 */       return WeakIdentityHashMap.this.size();
/*     */     }
/*     */ 
/*     */     public boolean contains(Object o) {
/* 753 */       return WeakIdentityHashMap.this.containsKey(o);
/*     */     }
/*     */ 
/*     */     public boolean remove(Object o) {
/* 757 */       if (WeakIdentityHashMap.this.containsKey(o)) {
/* 758 */         WeakIdentityHashMap.this.remove(o);
/* 759 */         return true;
/*     */       }
/*     */ 
/* 762 */       return false;
/*     */     }
/*     */ 
/*     */     public void clear() {
/* 766 */       WeakIdentityHashMap.this.clear();
/*     */     }
/*     */ 
/*     */     public Object[] toArray() {
/* 770 */       Collection c = new ArrayList(size());
/* 771 */       for (Iterator i = iterator(); i.hasNext(); )
/* 772 */         c.add(i.next());
/* 773 */       return c.toArray();
/*     */     }
/*     */ 
/*     */     public Object[] toArray(Object[] a) {
/* 777 */       Collection c = new ArrayList(size());
/* 778 */       for (Iterator i = iterator(); i.hasNext(); )
/* 779 */         c.add(i.next());
/* 780 */       return c.toArray(a);
/*     */     }
/*     */   }
/*     */ 
/*     */   private class EntryIterator extends WeakIdentityHashMap.HashIterator
/*     */   {
/*     */     private EntryIterator()
/*     */     {
/* 717 */       super();
/*     */     }
/* 719 */     public Object next() { return nextEntry();
/*     */     }
/*     */   }
/*     */ 
/*     */   private class KeyIterator extends WeakIdentityHashMap.HashIterator
/*     */   {
/*     */     private KeyIterator()
/*     */     {
/* 711 */       super();
/*     */     }
/* 713 */     public Object next() { return nextEntry().getKey();
/*     */     }
/*     */   }
/*     */ 
/*     */   private class ValueIterator extends WeakIdentityHashMap.HashIterator
/*     */   {
/*     */     private ValueIterator()
/*     */     {
/* 705 */       super();
/*     */     }
/* 707 */     public Object next() { return WeakIdentityHashMap.Entry.access$200(nextEntry());
/*     */     }
/*     */   }
/*     */ 
/*     */   private abstract class HashIterator
/*     */     implements Iterator
/*     */   {
/*     */     int index;
/* 636 */     WeakIdentityHashMap.Entry entry = null;
/* 637 */     WeakIdentityHashMap.Entry lastReturned = null;
/* 638 */     int expectedModCount = WeakIdentityHashMap.this.modCount;
/*     */ 
/* 644 */     Object nextKey = null;
/*     */ 
/* 650 */     Object currentKey = null;
/*     */ 
/*     */     HashIterator() {
/* 653 */       this.index = (WeakIdentityHashMap.this.size() != 0 ? WeakIdentityHashMap.this.table.length : 0);
/*     */     }
/*     */ 
/*     */     public boolean hasNext() {
/* 657 */       WeakIdentityHashMap.Entry[] t = WeakIdentityHashMap.this.table;
/*     */ 
/* 659 */       while (this.nextKey == null) {
/* 660 */         WeakIdentityHashMap.Entry e = this.entry;
/* 661 */         int i = this.index;
/* 662 */         while ((e == null) && (i > 0)) {
/* 663 */           i--; e = t[i];
/* 664 */         }this.entry = e;
/* 665 */         this.index = i;
/* 666 */         if (e == null) {
/* 667 */           this.currentKey = null;
/* 668 */           return false;
/*     */         }
/* 670 */         this.nextKey = e.get();
/* 671 */         if (this.nextKey == null)
/* 672 */           this.entry = WeakIdentityHashMap.Entry.access$100(this.entry);
/*     */       }
/* 674 */       return true;
/*     */     }
/*     */ 
/*     */     protected WeakIdentityHashMap.Entry nextEntry()
/*     */     {
/* 679 */       if (WeakIdentityHashMap.this.modCount != this.expectedModCount)
/* 680 */         throw new ConcurrentModificationException();
/* 681 */       if ((this.nextKey == null) && (!hasNext())) {
/* 682 */         throw new NoSuchElementException();
/*     */       }
/* 684 */       this.lastReturned = this.entry;
/* 685 */       this.entry = WeakIdentityHashMap.Entry.access$100(this.entry);
/* 686 */       this.currentKey = this.nextKey;
/* 687 */       this.nextKey = null;
/* 688 */       return this.lastReturned;
/*     */     }
/*     */ 
/*     */     public void remove() {
/* 692 */       if (this.lastReturned == null)
/* 693 */         throw new IllegalStateException();
/* 694 */       if (WeakIdentityHashMap.this.modCount != this.expectedModCount) {
/* 695 */         throw new ConcurrentModificationException();
/*     */       }
/* 697 */       WeakIdentityHashMap.this.remove(this.currentKey);
/* 698 */       this.expectedModCount = WeakIdentityHashMap.this.modCount;
/* 699 */       this.lastReturned = null;
/* 700 */       this.currentKey = null;
/*     */     }
/*     */   }
/*     */ 
/*     */   private static class Entry extends WeakReference
/*     */     implements Map.Entry
/*     */   {
/*     */     private Object value;
/*     */     private final int hash;
/*     */     private Entry next;
/*     */ 
/*     */     Entry(Object key, Object value, ReferenceQueue queue, int hash, Entry next)
/*     */     {
/* 587 */       super(queue);
/* 588 */       this.value = value;
/* 589 */       this.hash = hash;
/* 590 */       this.next = next;
/*     */     }
/*     */ 
/*     */     public Object getKey() {
/* 594 */       return WeakIdentityHashMap.access$300(get());
/*     */     }
/*     */ 
/*     */     public Object getValue() {
/* 598 */       return this.value;
/*     */     }
/*     */ 
/*     */     public Object setValue(Object newValue) {
/* 602 */       Object oldValue = this.value;
/* 603 */       this.value = newValue;
/* 604 */       return oldValue;
/*     */     }
/*     */ 
/*     */     public boolean equals(Object o) {
/* 608 */       if (!(o instanceof Map.Entry))
/* 609 */         return false;
/* 610 */       Map.Entry e = (Map.Entry)o;
/* 611 */       Object k1 = getKey();
/* 612 */       Object k2 = e.getKey();
/* 613 */       if (k1 == k2) {
/* 614 */         Object v1 = getValue();
/* 615 */         Object v2 = e.getValue();
/* 616 */         if ((v1 == v2) || ((v1 != null) && (v1.equals(v2))))
/* 617 */           return true;
/*     */       }
/* 619 */       return false;
/*     */     }
/*     */ 
/*     */     public int hashCode() {
/* 623 */       Object k = getKey();
/* 624 */       Object v = getValue();
/* 625 */       return (k == null ? 0 : System.identityHashCode(k)) ^ (v == null ? 0 : v.hashCode());
/*     */     }
/*     */ 
/*     */     public String toString()
/*     */     {
/* 630 */       return getKey() + "=" + getValue();
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.util.collection.WeakIdentityHashMap
 * JD-Core Version:    0.6.0
 */